<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-07-06
 ******************************************************************************/

namespace Fastmag\Sync\FastmagSetup;

use Fastmag\Sync\Model\Config;

/**
 * Class Installer
 *
 * Class handling job queue and triggers installations on Fastmag
 */
class Installer
{
    /** @var Config $config */
    protected $config;

    /**
     * Installer constructor.
     *
     * @param Config $config
     */
    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    /**
     * Returns "Import generic products" config
     *
     * @return bool
     */
    public function configGenericProducts()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_GENERIC_PRODUCTS);
    }

    /**
     * Returns "Parents per color" config
     *
     * @return bool
     */
    public function configParentPerColor()
    {
        return $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_PARENT_PER_COLOR);
    }

    /**
     * Get installation queries
     *
     * @return array
     */
    public function getInstallQueries()
    {
        $configGenericProducts = $this->configGenericProducts();
        $configParentPerColor = $this->configParentPerColor();

        $sql = ['triggers' => [], 'tables' => []];

        $sql['tables']['fmsync_job_queue'] = '
            CREATE TABLE IF NOT EXISTS fmsync_job_queue (
                id_job INT(10) NOT NULL AUTO_INCREMENT,
                identifiant VARCHAR(60) DEFAULT NULL,
                job VARCHAR(20) DEFAULT NULL,
                code_mag VARCHAR(30) DEFAULT NULL,
                date_creation DATETIME NOT NULL,
                date_maj DATETIME DEFAULT NULL,
                date_a_traiter DATETIME DEFAULT NULL,
                commentaire LONGTEXT DEFAULT NULL,
                priorite BIGINT(8) UNSIGNED NOT NULL DEFAULT 0,
                checksum VARCHAR(100) DEFAULT \'\',
                PRIMARY KEY(id_job),
                INDEX(job),
                INDEX(identifiant),
                INDEX(code_mag),
                INDEX(date_a_traiter),
                UNIQUE KEY index_job_queue_unique (job, identifiant, checksum)
            ) ENGINE=MYISAM DEFAULT CHARSET=utf8;';

        ///// Triggers

        // produitsfiches
        $sql['triggers']['trigger_fmsync_produitsfiches_update'] = '
            CREATE TRIGGER trigger_fmsync_produitsfiches_update
                AFTER UPDATE
                ON produitsfiches
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produitsfiches_update'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = NEW.BarCode,
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitsfiches_update\', \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produitsfiches_update'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, date_creation, date_maj)
                SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)),
                        \'ProductCreate\', 
                        JSON_OBJECT(\'trigger\', \'trigger_fmsync_produitsfiches_update\', \'type_reference\', \'color\'),
                        NOW(),
                        NOW()
                    FROM produits 
                    WHERE ProduitActif = 1 AND BarCode = NEW.BarCode
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produitsfiches_update'] .= '
            END;';

        // produitslangues
        $sql['triggers']['trigger_fmsync_produitslangues_update'] = '
            CREATE TRIGGER trigger_fmsync_produitslangues_update
                AFTER UPDATE
                ON produitslangues
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET identifiant   = NEW.BarCode,
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitslangues_update\',
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        // produits
        $sql['triggers']['trigger_fmsync_produits_update'] = '
            CREATE TRIGGER trigger_fmsync_produits_update
                AFTER UPDATE
                ON produits
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_update'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = NEW.BarCode,
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_update\', \'type_reference\', \'parent\'),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_update'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_update\', \'type_reference\', \'color\'),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produits_update'] .= '
            END;';

        // produitstarifs
        $sql['triggers']['trigger_fmsync_produitstarifs_insert'] = '
            CREATE TRIGGER trigger_fmsync_produitstarifs_insert
                AFTER INSERT
                ON produitstarifs
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produitstarifs_insert'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = NEW.BarCode,
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitstarifs_insert\',
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produitstarifs_insert'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitstarifs_insert\',
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produitstarifs_insert'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produitstarifs_update'] = '
            CREATE TRIGGER trigger_fmsync_produitstarifs_update
                AFTER UPDATE
                ON produitstarifs
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produitstarifs_update'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = NEW.BarCode,
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitstarifs_update\',
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produitstarifs_update'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitstarifs_update\',
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produitstarifs_update'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produitstarifs_delete'] = '
            CREATE TRIGGER trigger_fmsync_produitstarifs_delete
                AFTER DELETE
                ON produitstarifs
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produitstarifs_delete'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = OLD.BarCode,
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitstarifs_delete\',
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produitstarifs_delete'] .= '
                INSERT INTO fmsync_job_queue
                SET identifiant   = CONCAT(OLD.BarCode, \'-\', TRIM(OLD.Couleur)),
                    job           = \'ProductCreate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produitstarifs_delete\',
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produitstarifs_delete'] .= '
            END;';

        // prixremise
        $sql['triggers']['trigger_fmsync_produits_promo_insert'] = '
            CREATE TRIGGER trigger_fmsync_produits_promo_insert
                AFTER INSERT
                ON prixremise
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_promo_insert'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT BarCode, 
                        \'ProductSpecialPrice\',
                        JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promo_insert\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'parent\'
                        ),
                        80,
                        NOW(),
                        NOW()
                    FROM produitsfiches 
                    WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = NEW.BarCode
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT BarCode,
                            \'ProductSpecialPrice\', 
                            JSON_OBJECT(
                                \'trigger\', \'trigger_fmsync_produits_promo_insert\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_debut\', NEW.DateDebut,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'parent\'
                            ),
                            80 , 
                            NOW(), 
                            NOW(),
                            NEW.DateDebut,
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produitsfiches 
                        WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = NEW.BarCode
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT BarCode, 
                            \'ProductSpecialPrice\', 
                            CONCAT_WS(
                                \'trigger\', \'trigger_fmsync_produits_promo_insert\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_fin\', NEW.DateFin,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'parent\'
                            ),
                            80 , 
                            NOW(),
                            NOW(),
                            CONCAT(NEW.DateFin, \' 23:59:59\'),
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produitsfiches 
                        WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = NEW.BarCode 
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_promo_insert'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)),
                        \'ProductSpecialPrice\',
                        JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promo_insert\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'color\'
                        ),
                        80,
                        NOW(),
                        NOW()
                    FROM produits 
                    WHERE ProduitActif = 1 AND BarCode = NEW.BarCode        
                ON DUPLICATE KEY UPDATE date_maj = NOW();
                        
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)), 
                            \'ProductSpecialPrice\',
                            JSON_OBJECT(
                                \'trigger\', \'trigger_fmsync_produits_promo_insert\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_debut\', NEW.DateDebut,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'color\'
                            ),
                            80, 
                            NOW(),
                            NOW(),
                            NEW.DateDebut,
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produits 
                        WHERE ProduitActif = 1 AND BarCode = NEW.BarCode        
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
        
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)), 
                            \'ProductSpecialPrice\', 
                            JSON_OBJECT(
                                \'trigger\', \'trigger_fmsync_produits_promo_insert\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_fin\', NEW.DateFin,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'color\'
                            ),
                            80, 
                            NOW(),
                            NOW(),
                            CONCAT(NEW.DateFin, \' 23:59:59\'),
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produits 
                        WHERE ProduitActif = 1 AND BarCode = NEW.BarCode        
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        }

        $sql['triggers']['trigger_fmsync_produits_promo_insert'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produits_promo_update'] = '
            CREATE TRIGGER trigger_fmsync_produits_promo_update
                AFTER UPDATE
                ON prixremise
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_promo_update'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT BarCode,
                        \'ProductSpecialPrice\',
                        JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promo_update\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'parent\'
                        ),
                        80,
                        NOW(), 
                        NOW()
                    FROM produitsfiches 
                    WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = NEW.BarCode
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT BarCode,
                            \'ProductSpecialPrice\', 
                            CONCAT_WS(
                                \'trigger\', \'trigger_fmsync_produits_promo_update\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_debut\', NEW.DateDebut,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'parent\'
                            ),
                            80 , 
                            NOW(), 
                            NOW(),
                            NEW.DateDebut,
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produitsfiches 
                        WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = NEW.BarCode
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT BarCode, 
                            \'ProductSpecialPrice\', 
                            CONCAT_WS(
                                \'trigger\', \'trigger_fmsync_produits_promo_update\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_fin\', NEW.DateFin,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'parent\'
                            ),
                            80, 
                            NOW(),
                            NOW(),
                            CONCAT(NEW.DateFin, \' 23:59:59\'),
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produitsfiches 
                        WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = NEW.BarCode 
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_promo_update'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)),
                        \'ProductSpecialPrice\',
                        JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promo_update\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'color\'
                        ),
                        80,
                        NOW(),
                        NOW()
                    FROM produits 
                    WHERE ProduitActif = 1 AND BarCode = NEW.BarCode        
                ON DUPLICATE KEY UPDATE date_maj = NOW();
                        
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)), 
                            \'ProductSpecialPrice\',
                            JSON_OBJECT(
                                \'trigger\', \'trigger_fmsync_produits_promo_update\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_debut\', NEW.DateDebut,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'color\'
                            ),
                            80, 
                            NOW(),
                            NOW(),
                            NEW.DateDebut,
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produits 
                        WHERE ProduitActif = 1 AND BarCode = NEW.BarCode        
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
        
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                        (identifiant, job, commentaire, priorite, date_creation, date_maj, date_a_traiter, code_mag, checksum)
                    SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)), 
                            \'ProductSpecialPrice\', 
                            JSON_OBJECT(
                                \'trigger\', \'trigger_fmsync_produits_promo_update\',
                                \'code_magasin\', NEW.CodeMag,
                                \'date_fin\', NEW.DateFin,
                                \'prix\', NEW.Prix,
                                \'remise\', NEW.Remise,
                                \'type_reference\', \'color\'
                            ),
                            80, 
                            NOW(),
                            NOW(),
                            CONCAT(NEW.DateFin, \' 23:59:59\'),
                            NEW.CodeMag,
                            CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                        FROM produits 
                        WHERE ProduitActif = 1 AND BarCode = NEW.BarCode        
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        }

        $sql['triggers']['trigger_fmsync_produits_promo_update'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produits_promo_delete'] = '
            CREATE TRIGGER trigger_fmsync_produits_promo_delete
                AFTER DELETE
                ON prixremise
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_promo_delete'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT BarCode,
                        \'ProductSpecialPrice\', 
                        JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_promo_delete\', \'type_reference\', \'parent\'),
                        80,
                        NOW(),
                        NOW() 
                    FROM produitsfiches 
                    WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = OLD.BarCode
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_promo_delete'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)),
                        \'ProductSpecialPrice\', 
                        JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_promo_delete\', \'type_reference\', \'color\'),
                        80,
                        NOW(),
                        NOW() 
                    FROM produits 
                    WHERE ProduitActif = 1 AND BarCode = OLD.BarCode        
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produits_promo_delete'] .= '
            END;';

        // prixremisetc
        $sql['triggers']['trigger_fmsync_produits_promotc_insert'] = '
            CREATE TRIGGER trigger_fmsync_produits_promotc_insert
                AFTER INSERT
                ON prixremisetc
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_promotc_insert'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite    = 80,
                    identifiant = NEW.BarCode,
                    job         = \'ProductSpecialPrice\',
                    code_mag    = NEW.CodeMag,
                    commentaire = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_promotc_insert\',
                        \'code_magasin\', NEW.CodeMag,
                        \'date_debut\', NEW.DateDebut,
                        \'date_fin\', NEW.DateFin,
                        \'prix\', NEW.Prix,
                        \'remise\', NEW.Remise,
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = NEW.BarCode,
                        date_a_traiter = NEW.DateDebut,
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_insert\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'parent\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = NEW.BarCode,
                        date_a_traiter = CONCAT(NEW.DateFin, \' 23:59:59\'),
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_insert\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'parent\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_promotc_insert'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite    = 80,
                    identifiant = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job         = \'ProductSpecialPrice\',
                    code_mag    = NEW.CodeMag,
                    commentaire = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_promotc_insert\',
                        \'code_magasin\', NEW.CodeMag,
                        \'date_debut\', NEW.DateDebut,
                        \'date_fin\', NEW.DateFin,
                        \'prix\', NEW.Prix,
                        \'remise\', NEW.Remise,
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                        date_a_traiter = NEW.DateDebut,
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_insert\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'color\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                        date_a_traiter = CONCAT(NEW.DateFin, \' 23:59:59\'),
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_insert\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'color\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        }

        $sql['triggers']['trigger_fmsync_produits_promotc_insert'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produits_promotc_update'] = '
            CREATE TRIGGER trigger_fmsync_produits_promotc_update
                AFTER UPDATE
                ON prixremisetc
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_promotc_update'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite    = 80,
                    identifiant = NEW.BarCode,
                    job         = \'ProductSpecialPrice\',
                    code_mag    = NEW.CodeMag,
                    commentaire = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_promotc_update\',
                        \'code_magasin\', NEW.CodeMag,
                        \'date_debut\', NEW.DateDebut,
                        \'date_fin\', NEW.DateFin,
                        \'prix\', NEW.Prix,
                        \'remise\', NEW.Remise,
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = NEW.BarCode,
                        date_a_traiter = NEW.DateDebut,
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_update\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'parent\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = NEW.BarCode,
                        date_a_traiter = CONCAT(NEW.DateFin, \' 23:59:59\'),
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_update\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'parent\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_promotc_update'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite    = 80,
                    identifiant = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job         = \'ProductSpecialPrice\',
                    code_mag    = NEW.CodeMag,
                    commentaire = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_promotc_update\',
                        \'code_magasin\', NEW.CodeMag,
                        \'date_debut\', NEW.DateDebut,
                        \'date_fin\', NEW.DateFin,
                        \'prix\', NEW.Prix,
                        \'remise\', NEW.Remise,
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            
                IF NEW.DateDebut IS NOT NULL AND NEW.DateDebut > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                        date_a_traiter = NEW.DateDebut,
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_update\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_debut\', NEW.DateDebut,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'color\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateDebut, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            
                IF NEW.DateFin IS NOT NULL AND NEW.DateFin > NOW() THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant    = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                        date_a_traiter = CONCAT(NEW.DateFin, \' 23:59:59\'),
                        job            = \'ProductSpecialPrice\',
                        code_mag       = NEW.CodeMag,
                        commentaire    = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_produits_promotc_update\',
                            \'code_magasin\', NEW.CodeMag,
                            \'date_fin\', NEW.DateFin,
                            \'prix\', NEW.Prix,
                            \'remise\', NEW.Remise,
                            \'type_reference\', \'color\'
                        ),
                        date_creation  = NOW(),
                        date_maj       = NOW(),
                        checksum       = CONCAT(NEW.BarCode, NEW.DateFin, NEW.CodeMag, \'ProductSpecialPrice\')
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;';
        }

        $sql['triggers']['trigger_fmsync_produits_promotc_update'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produits_promotc_delete'] = '
            CREATE TRIGGER trigger_fmsync_produits_promotc_delete
                AFTER DELETE
                ON prixremisetc
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_promotc_delete'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT BarCode,
                        \'ProductSpecialPrice\', 
                        JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_promotc_delete\', \'type_reference\', \'parent\'),
                        80,
                        NOW(),
                        NOW() 
                    FROM produitsfiches 
                    WHERE Actif = 1 AND VisibleWeb = 1 AND BarCode = OLD.BarCode
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_promotc_delete'] .= '
                INSERT INTO fmsync_job_queue (identifiant, job, commentaire, priorite, date_creation, date_maj)
                SELECT CONCAT(BarCode, \'-\', TRIM(Couleur)),
                        \'ProductSpecialPrice\', 
                        JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_promotc_delete\', \'type_reference\', \'color\'),
                        80,
                        NOW(),
                        NOW() 
                    FROM produits 
                    WHERE ProduitActif = 1 AND BarCode = OLD.BarCode        
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produits_promotc_delete'] .= '
            END;';

        // produitsprixqte
        $sql['triggers']['trigger_fmsync_produits_prixqte_insert'] = '
            CREATE TRIGGER trigger_fmsync_produits_prixqte_insert
                AFTER INSERT
                ON produitsprixqte
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET priorite      = 10,
                    identifiant   = NEW.BarCode,
                    job           = \'ProductUpdate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_prixqte_insert\',
                        \'prix\', NEW.Prix,
                        \'remise\', NEW.Remise
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        $sql['triggers']['trigger_fmsync_produits_prixqte_update'] = '
            CREATE TRIGGER trigger_fmsync_produits_prixqte_update
                AFTER UPDATE
                ON produitsprixqte
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET priorite      = 10,
                    identifiant   = NEW.BarCode,
                    job           = \'ProductUpdate\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_prixqte_update\',
                        \'prix\', NEW.Prix,
                        \'remise\', NEW.Remise
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        $sql['triggers']['trigger_fmsync_produits_prixqte_delete'] = '
            CREATE TRIGGER trigger_fmsync_produits_prixqte_delete
                AFTER DELETE
                ON produitsprixqte
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET priorite      = 10,
                    identifiant   = OLD.BarCode,
                    job           = \'ProductUpdate\',
                    commentaire   = JSON_OBJECT(\'trigger\', \'trigger_fmsync_produits_prixqte_delete\'),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        // imagestocks
        $sql['triggers']['trigger_fmsync_produits_image_stock_insert'] = '
            CREATE TRIGGER trigger_fmsync_produits_image_stock_insert
                AFTER INSERT
                ON imagestocks
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_image_stock_insert'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite      = 100,
                    identifiant   = NEW.BarCode,
                    job           = \'ProductQuantity\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_image_stock_insert\',
                        \'code_magasin\', NEW.CodeMag,
                        \'couleur\', NEW.Couleur,
                        \'taille\', NEW.Taille,
                        \'stock\', NEW.Stock,
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_image_stock_insert'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite      = 100,
                    identifiant   = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job           = \'ProductQuantity\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_image_stock_insert\',
                        \'code_magasin\', NEW.CodeMag,
                        \'couleur\', NEW.Couleur,
                        \'taille\', NEW.Taille,
                        \'stock\', NEW.Stock,
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produits_image_stock_insert'] .= '
            END;';

        $sql['triggers']['trigger_fmsync_produits_image_stock_update'] = '
            CREATE TRIGGER trigger_fmsync_produits_image_stock_update
                AFTER UPDATE
                ON imagestocks
                FOR EACH ROW
            BEGIN';

        if (!$configParentPerColor || ($configParentPerColor && $configGenericProducts)) {
            $sql['triggers']['trigger_fmsync_produits_image_stock_update'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite      = 100,
                    identifiant   = NEW.BarCode,
                    job           = \'ProductQuantity\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_image_stock_update\',
                        \'code_magasin\', NEW.CodeMag,
                        \'couleur\', NEW.Couleur,
                        \'taille\', NEW.Taille,
                        \'stock\', NEW.Stock,
                        \'type_reference\', \'parent\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        } elseif ($configParentPerColor) {
            $sql['triggers']['trigger_fmsync_produits_image_stock_update'] .= '
                INSERT INTO fmsync_job_queue
                SET priorite      = 100,
                    identifiant   = CONCAT(NEW.BarCode, \'-\', TRIM(NEW.Couleur)),
                    job           = \'ProductQuantity\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_image_stock_update\',
                        \'code_magasin\', NEW.CodeMag,
                        \'couleur\', NEW.Couleur,
                        \'taille\', NEW.Taille,
                        \'stock\', NEW.Stock,
                        \'type_reference\', \'color\'
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();';
        }

        $sql['triggers']['trigger_fmsync_produits_image_stock_update'] .= '
            END;';

        // ventes
        $sql['triggers']['trigger_fmsync_vente_update'] = '
            CREATE TRIGGER trigger_fmsync_vente_update
                AFTER UPDATE
                ON ventes
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET priorite      = 20,
                    identifiant   = NEW.Vente,
                    job           = \'OrderStatus\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_vente_update\',
                        \'vente\', NEW.Vente,
                        \'code_magasin\', NEW.CodeMag,
                        \'nature\', NEW.Nature
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        // lignes
        $sql['triggers']['trigger_fmsync_produits_vente_insert'] = '
            CREATE TRIGGER trigger_fmsync_produits_vente_insert
                AFTER INSERT
                ON lignes
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET priorite      = 20,
                    identifiant   = NEW.BarCode,
                    job           = \'ProductQuantity\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_vente_insert\',
                        \'vente\', NEW.Vente,
                        \'id_stock\', NEW.StockID,
                        \'code_magasin\', NEW.CodeMag,
                        \'couleur\', NEW.Couleur,
                        \'taille\', NEW.Taille,
                        \'quantite\', NEW.Quantite,
                        \'motif\', NEW.Motif
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        $sql['triggers']['trigger_fmsync_produits_vente_update'] = '
            CREATE TRIGGER trigger_fmsync_produits_vente_update
                AFTER UPDATE
                ON lignes
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET priorite      = 100,
                    identifiant   = NEW.BarCode,
                    job           = \'ProductQuantity\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_produits_vente_update\',
                        \'vente\', NEW.Vente,
                        \'id_stock:\', NEW.StockID,
                        \'code_magasin\', NEW.CodeMag,
                        \'couleur\', NEW.Couleur,
                        \'taille\', NEW.Taille,
                        \'quantite\', NEW.Quantite,
                        \'motif\', NEW.Motif
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        // clients
        $sql['triggers']['trigger_fmsync_client_insert'] = '
            CREATE TRIGGER trigger_fmsync_client_insert
                AFTER INSERT
                ON clients
                FOR EACH ROW
            BEGIN
                IF ((@TRIGGER_CLIENT_INSERT_OFF = FALSE OR @TRIGGER_CLIENT_INSERT_OFF IS NULL)) THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant   = NEW.Client,
                        job           = \'CustomerCreate\',
                        commentaire   = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_client_insert\',
                            \'client\', NEW.Client,
                            \'email\', NEW.email
                        ),
                        date_creation = NOW(),
                        date_maj      = NOW()
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            END;';

        $sql['triggers']['trigger_fmsync_client_update'] = '
            CREATE TRIGGER trigger_fmsync_client_update
                AFTER UPDATE
                ON clients
                FOR EACH ROW
            BEGIN
                IF ((@TRIGGER_CLIENT_UPDATE_OFF = FALSE OR @TRIGGER_CLIENT_UPDATE_OFF IS NULL)) THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant   = NEW.Client,
                        job           = \'CustomerUpdate\',
                        commentaire   = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_client_update\',
                            \'client\', NEW.Client,
                            \'email\', NEW.email
                        ),
                        date_creation = NOW(),
                        date_maj      = NOW()
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            END;';

        // clientadresse
        $sql['triggers']['trigger_fmsync_client_addr_liv_insert'] = '
            CREATE TRIGGER trigger_fmsync_client_addr_liv_insert
                AFTER INSERT
                ON clientadresse
                FOR EACH ROW
            BEGIN
                IF ((@TRIGGER_CLIENTADR_INSERT_OFF = FALSE OR @TRIGGER_CLIENTADR_INSERT_OFF IS NULL)) THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant   = CONCAT_WS(\';\', NEW.Client, NEW.AdrLivraison),
                        job           = \'AddressCreate\',
                        commentaire   = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_client_addr_liv_insert\',
                            \'client\', NEW.Client,
                            \'adresse_livraison\', NEW.AdrLivraison
                        ),
                        date_creation = NOW(),
                        date_maj      = NOW()
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            END;';

        $sql['triggers']['trigger_fmsync_client_addr_liv_update'] = '
            CREATE TRIGGER trigger_fmsync_client_addr_liv_update
                AFTER UPDATE
                ON clientadresse
                FOR EACH ROW
            BEGIN
                IF ((@TRIGGER_CLIENTADR_UPDATE_OFF = FALSE OR @TRIGGER_CLIENTADR_UPDATE_OFF IS NULL)) THEN
                    INSERT INTO fmsync_job_queue
                    SET identifiant   = CONCAT_WS(\';\', NEW.Client, NEW.AdrLivraison),
                        job           = \'AddressUpdate\',
                        commentaire   = JSON_OBJECT(
                            \'trigger\', \'trigger_fmsync_client_addr_liv_update\',
                            \'client\', NEW.Client,
                            \'adresse_livraison\', NEW.AdrLivraison
                        ),
                        date_creation = NOW(),
                        date_maj      = NOW()
                    ON DUPLICATE KEY UPDATE date_maj = NOW();
                END IF;
            END;';

        $sql['triggers']['trigger_fmsync_client_addr_liv_delete'] = '
            CREATE TRIGGER trigger_fmsync_client_addr_liv_delete
                AFTER DELETE
                ON clientadresse
                FOR EACH ROW
            BEGIN
                INSERT INTO fmsync_job_queue
                SET identifiant   = CONCAT_WS(\';\', OLD.Client, OLD.AdrLivraison),
                    job           = \'AddressDelete\',
                    commentaire   = JSON_OBJECT(
                        \'trigger\', \'trigger_fmsync_client_addr_liv_delete\',
                        \'client\', OLD.Client,
                        \'adresse_livraison:\', OLD.AdrLivraison
                    ),
                    date_creation = NOW(),
                    date_maj      = NOW()
                ON DUPLICATE KEY UPDATE date_maj = NOW();
            END;';

        return $sql;
    }
}
