<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-17
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process;

use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;

/**
 * Class Manager
 *
 * Parent class of the process manager (remote sync, to magento and to fastmag)
 */
class Manager
{
    /** @var Logger $logger */
    protected $logger;

    /** @var Config $config */
    protected $config;

    /**
     * Job queue manager constructor.
     *
     * @param Logger $logger
     * @param Config $config
     */
    public function __construct(
        Logger $logger,
        Config $config
    ) {
        $this->logger = $logger;
        $this->config = $config;
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites()
    {
        $result = true;

        if (!$this->isSyncEnabled()) {
            throw new ProcessException('Synchronization is currently disabled. Remote sync can not be done.');
        }

        if ($this->getWebsiteId() === null) {
            throw new ProcessException('Website ID not set in config.');
        }

        if ($this->getSyncLimit() <= 0) {
            throw new ProcessException('Process queue limit size not set in config.');
        }

        return $result;
    }

    /**
     * Check if global synchronization is enabled
     *
     * @return bool
     */
    protected function isSyncEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_JOBQUEUE_SYNC_ENABLED);
    }

    /**
     * Check if website ID for synchonization is set in config
     *
     * @return string
     */
    protected function getWebsiteId()
    {
        return $this->config->getValue(Config::XML_PATH_JOBQUEUE_SYNC_WEBSITE_ID);
    }

    /**
     * Get sync limit
     *
     * @return int
     */
    protected function getSyncLimit()
    {
        return (int)$this->config->getValue(Config::XML_PATH_JOBQUEUE_ADVANCED_JOBS_LIMIT);
    }
}
