<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-04-29
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process;

use DateTime;
use DateTimeZone;
use Exception;
use Fastmag\Sync\Logger\Logger;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\Pdo\Mysql;

/**
 * Class Worker
 *
 * Class handling basic operation on the workers, and inherited to hydration and integration workers
 */
abstract class Worker
{
    /** @var string */
    public const TIMEZONE_FASTMAG = 'Europe/Paris';

    /** @var string */
    public const TIMEZONE_UTC = 'UTC';

    /** @var string $code */
    protected $code;

    /** @var Logger $logger */
    protected $logger;

    /** @var ResourceConnection $resourceConnection */
    protected $resourceConnection;

    /**
     * Worker constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection
    ) {
        $this->logger = $logger;
        $this->resourceConnection = $resourceConnection;
    }

    /**
     * @return string
     */
    public function getCode()
    {
        return $this->code;
    }

    /**
     * Main method
     *
     * @return void
     */
    abstract public function run();

    /**
     * Get manager last execution date
     *
     * Returns date as string if there is an Exception threw by DateTime constructor
     *
     * @return DateTime|string
     */
    public function getLastExecutionDate()
    {
        $select = $this->resourceConnection->getConnection()
            ->select()
            ->from('fastmag_sync_job_last_execution', 'last_executed_at')
            ->where('job_code = \'' . $this->getCode() . '\'');

        $value = $this->resourceConnection->getConnection()->fetchOne($select);

        try {
            if ($value === false) {
                $result = new DateTime('1970-01-01');
            } else {
                $result = new DateTime($value);
            }
        } catch (Exception $e) {
            $result = $value;
        }

        return $result;
    }

    /**
     * Set manager last execution date
     *
     * @param Datetime|string $date
     *
     * @return bool
     */
    public function setLastExecutionDate($date = null)
    {
        if ($date === null) {
            try {
                $date = new DateTime();
            } catch (Exception $e) {
                $date = date(Mysql::DATETIME_FORMAT);
            }
        }

        if (get_class($date) === 'DateTime') {
            $date = $date->format(Mysql::DATETIME_FORMAT);
        }

        $data = [
            'job_code'         => $this->getCode(),
            'last_executed_at' => $date
        ];

        $affectedRows = $this->resourceConnection->getConnection()
            ->insertOnDuplicate(
                'fastmag_sync_job_last_execution',
                $data,
                ['last_executed_at']
            );

        return ($affectedRows === 1);
    }

    /**
     * Returns string if there is an exception threw by DateTime constructor
     *
     * @param string $date
     *
     * @return DateTime|string
     */
    protected function getUtcDateTimeFromFastmag($date)
    {
        try {
            $dateTime = new DateTime($date, new DateTimeZone(self::TIMEZONE_FASTMAG));
            $dateTime->setTimezone(new DateTimeZone(self::TIMEZONE_UTC));
        } catch (Exception $e) {
            $dateTime = $date;
        }

        return $dateTime;
    }
}
