<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-26
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomerInterface;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\CustomerRepository as SyncedCustomerRepository;
use Fastmag\Sync\Model\Jobqueue\ToFastmag as Job;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\Collection;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Customer
 *
 * Hydration class used for inserting or updating customers from Magento to Fastmag
 */
abstract class Customer extends Hydration
{
    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var CustomerRepositoryInterface $customerRepository */
    protected $customerRepository;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var int[] $customersIds */
    protected $customersIds;

    /** @var array $hydratedData */
    protected $hydratedData;

    /**
     * Customer constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param Collection                  $jobs
     * @param Config                      $config
     * @param SyncedCustomerRepository    $syncedCustomerRepository
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     * @param CustomerRepositoryInterface $customerRepository
     * @param JobRepository               $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Collection $jobs,
        Config $config,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepositoryInterface $customerRepository,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $jobs, $config);

        $this->syncedCustomerRepository = $syncedCustomerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->customerRepository = $customerRepository;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($results as $customerId => $row) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    $job->setHydratedData($row);
                }
            }

            foreach ($this->jobs as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $this->customersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);
        $this->hydratedData = [];

        foreach ($this->customersIds as $customerId) {
            $this->hydratedData[$customerId] = [];
        }

        $this->getSyncedCustomers();
        $this->getMagentoCustomersData();

        return $this->hydratedData;
    }

    /**
     * Get synced Fastmag ids for the current jobs
     *
     * @return void
     */
    protected function getSyncedCustomers()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SyncedCustomerInterface::MAGENTO_CUSTOMER_ID, $this->customersIds, 'in')
            ->create();

        $syncedCustomerList = $this->syncedCustomerRepository->getList($searchCriteria);

        foreach ($syncedCustomerList->getItems() as $syncedCustomer) {
            $this->hydratedData[$syncedCustomer->getMagentoCustomerId()]['fastmag_id']
                = $syncedCustomer->getFastmagCustomerId();
        }
    }

    /**
     * Get Magento data for the current jobs
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getMagentoCustomersData()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $this->customersIds, 'in')
            ->create();

        try {
            $customersList = $this->customerRepository->getList($searchCriteria);

            foreach ($customersList->getItems() as $customer) {
                $customerData = [
                    'gender'     => $customer->getGender(),
                    'lastname'   => $customer->getLastname(),
                    'firstname'  => $customer->getFirstname(),
                    'email'      => $customer->getEmail(),
                    'created_at' => $customer->getCreatedAt(),
                    'website_id' => $customer->getWebsiteId(),
                    'store_id'   => $customer->getStoreId(),
                    'dob'        => $customer->getDob(),
                    'group_id'   => $customer->getGroupId()
                ];

                foreach ($customerData as $field => $value) {
                    $this->hydratedData[$customer->getId()][$field] = $value;
                }
            }
        } catch (Exception $e) {
            throw new ProcessException(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                implode(', ', $this->customersIds)
            );
        }
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerId
     *
     * @return Job
     */
    protected function getJob($customerId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerId);
    }
}
