<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-17
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration\Customer;

use Exception;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Customer\Address\Constants;
use Fastmag\Sync\Model\Jobqueue\ToFastmag as Job;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\Collection;
use Magento\Customer\Api\AddressRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\AddressInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Sales\Api\Data\OrderAddressInterface;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
class Address extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_address';

    /** @var OrderRepositoryInterface $orderRepository */
    protected $orderRepository;

    /** @var CustomerRepositoryInterface $customerRepository */
    protected $customerRepository;

    /** @var AddressRepositoryInterface $addressRepository */
    protected $addressRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /**
     * Address constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param Collection                  $jobs
     * @param Config                      $config
     * @param OrderRepositoryInterface    $orderRepository
     * @param CustomerRepositoryInterface $customerRepository
     * @param AddressRepositoryInterface  $addressRepository
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     * @param SortOrderBuilder            $sortOrderBuilder
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Collection $jobs,
        Config $config,
        OrderRepositoryInterface $orderRepository,
        CustomerRepositoryInterface $customerRepository,
        AddressRepositoryInterface $addressRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder
    ) {
        parent::__construct($logger, $resourceConnection, $jobs, $config);

        $this->orderRepository = $orderRepository;
        $this->customerRepository = $customerRepository;
        $this->addressRepository = $addressRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($results as $customerId => $row) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    $customerData = $job->getHydratedData();
                    $customerData['addresses'] = $row;
                    $job->setHydratedData($customerData);
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromMagento()
    {
        $customersIds = $this->getCustomersIds();
        $result = [];

        foreach ($customersIds as $customerId) {
            /** @var Order $lastOrder */
            $lastOrder = $this->getLastOrder($customerId);

            $billingAddressData = $this->getAddressData($customerId, $lastOrder, 'billing');
            $shippingAddressData = $this->getAddressData($customerId, $lastOrder, 'shipping');

            $hydratedData = [
                'billing_address'  => $billingAddressData,
                'shipping_address' => $shippingAddressData
            ];

            $result[$customerId] = $hydratedData;
        }

        return $result;
    }

    /**
     * Returns the list of the customers IDs, if the jobs is linked to customers.
     *
     * @return int[]
     */
    protected function getCustomersIds()
    {
        return $this->jobs->getColumnValues(Job::CONTENT_ID);
    }

    /**
     * Get last order ID made by the customer (if the customer has made one)
     *
     * @param int $customerId
     *
     * @return OrderInterface|bool
     */
    protected function getLastOrder($customerId)
    {
        $result = false;

        $sortOrder = $this->sortOrderBuilder->setField(OrderInterface::CREATED_AT)
            ->setDescendingDirection()
            ->create();

        $searchCriteria = $this->searchCriteriaBuilder->addFilter(OrderInterface::CUSTOMER_ID, $customerId)
            ->addSortOrder($sortOrder)
            ->setPageSize(1)
            ->create();

        $orderList = $this->orderRepository->getList($searchCriteria);

        if ($orderList->getTotalCount() === 1) {
            $result = $orderList->getItems()[0];
        }

        return $result;
    }

    /**
     * Get customer's last order addresses or default addresses if there is no order
     *
     * @param int    $customerId
     * @param Order  $lastOrder
     * @param string $type
     *
     * @return array
     */
    protected function getAddressData($customerId, $lastOrder, $type = 'billing')
    {
        $addressData = null;
        $addressObject = null;

        if ($lastOrder !== false) {
            if ($type === 'billing') {
                $addressObject = $lastOrder->getBillingAddress();
            } else {
                $addressObject = $lastOrder->getShippingAddress();
            }
        } else {
            try {
                $customer = $this->customerRepository->getById($customerId);
            } catch (Exception $e) {
                $customer = null;
            }

            if ($customer !== null) {
                if ($type === 'billing') {
                    $addressId = $customer->getDefaultBilling();
                } else {
                    $addressId = $customer->getDefaultShipping();
                }

                if ($addressId) {
                    try {
                        $addressObject = $this->addressRepository->getById($addressId);
                    } catch (Exception $e) {
                        $addressObject = null;
                    }
                }
            }
        }

        if ($addressObject !== null) {
            $addressAlias = $this->getAddressAlias($addressObject, $lastOrder, $customerId, $type);

            $addressData = [
                'alias'     => $addressAlias,
                'lastname'  => $addressObject->getLastname(),
                'firstname' => $addressObject->getFirstname(),
                'company'   => $addressObject->getCompany(),
                'street1'   => $addressObject->getStreet()[0],
                'postcode'  => $addressObject->getPostcode(),
                'city'      => $addressObject->getCity(),
                'phone'     => $addressObject->getTelephone(),
                'country'   => $addressObject->getCountryId()
            ];

            if (count($addressObject->getStreet()) > 1) {
                $addressData['street2'] = $addressObject->getStreet()[1];
            }

            if ($type === 'shipping') {
                $addressData['archive'] = (bool)$lastOrder;
            }
        }

        return $addressData;
    }

    /**
     * Defines the address alias
     *
     * @param AddressInterface|OrderAddressInterface $shippingAddress
     * @param Order|null                             $lastOrder
     * @param int                                    $customerId
     * @param string                                 $addressType
     *
     * @return string
     */
    protected function getAddressAlias($shippingAddress, $lastOrder, $customerId, $addressType = 'billing')
    {
        $result = '';

        if ($addressType === 'billing') {
            $result = ($lastOrder !== false ? $lastOrder->getId() . '_INVOICE' : 'INVOICE');
        } elseif ($addressType === 'shipping') {
            $shippingAddressAlias = $shippingAddress->getCustomAttribute(Constants::ALIAS_ATTRIBUTE_CODE);
            if ($shippingAddressAlias !== null) {
                $shippingAddressAlias = $shippingAddressAlias->getValue();
            }

            if ($shippingAddressAlias === '.' || $shippingAddressAlias === '') {
                $shippingAddressAlias = $customerId;
            }

            if ($lastOrder->getId()) {
                $shippingAddressAlias = $lastOrder->getId() . '_' . $shippingAddressAlias;
            }

            $result = Text::upper(substr(Text::sanitize(Text::removeAccents($shippingAddressAlias)), 0, 20));
        }

        return $result;
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerId
     *
     * @return Job
     */
    protected function getJob($customerId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerId);
    }
}
