<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-26
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration\Customer;

use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\CustomerRepository as SyncedCustomerRepository;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration\Customer;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration\Customer\Address as AddressHydrationWorker;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\Collection;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Newsletter\Model\Subscriber;

/**
 * Class Save
 *
 * Hydration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_save';

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var AddressHydrationWorker $addressHydrationWorker */
    protected $addressHydrationWorker;

    /**
     * Customer constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param Collection                  $jobs
     * @param Config                      $config
     * @param SyncedCustomerRepository    $syncedCustomerRepository
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     * @param CustomerRepositoryInterface $customerRepository
     * @param JobRepository               $jobRepository
     * @param Subscriber                  $subscriber
     * @param AddressHydrationWorker      $addressHydrationWorker
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Collection $jobs,
        Config $config,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepositoryInterface $customerRepository,
        JobRepository $jobRepository,
        Subscriber $subscriber,
        AddressHydrationWorker $addressHydrationWorker
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $jobs,
            $config,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $jobRepository
        );

        $this->subscriber = $subscriber;
        $this->addressHydrationWorker = $addressHydrationWorker;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($results as $customerId => $row) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    $job->setHydratedData($row);
                }
            }

            $this->addressHydrationWorker->setJobs($this->jobs)->run();

            foreach ($this->jobs as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            }
        }
    }

    /**
     * Get Magento data for the current jobs
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getMagentoCustomersData()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $this->customersIds, 'in')
            ->create();

        try {
            $customersList = $this->customerRepository->getList($searchCriteria);

            foreach ($customersList->getItems() as $customer) {
                $customerData = [
                    'gender'     => $customer->getGender(),
                    'lastname'   => $customer->getLastname(),
                    'firstname'  => $customer->getFirstname(),
                    'email'      => $customer->getEmail(),
                    'created_at' => $customer->getCreatedAt(),
                    'website_id' => $customer->getWebsiteId(),
                    'store_id'   => $customer->getStoreId(),
                    'dob'        => $customer->getDob(),
                    'group_id'   => $customer->getGroupId()
                ];

                $checkSubscriber = $this->subscriber->loadByCustomerId($customer->getId());
                $customerData['newsletter'] = $checkSubscriber->isSubscribed();

                foreach ($customerData as $field => $value) {
                    $this->hydratedData[$customer->getId()][$field] = $value;
                }
            }
        } catch (LocalizedException $e) {
            throw new ProcessException(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                implode(', ', $this->customersIds)
            );
        }
    }
}
