<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-17
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer\Address;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToFastmag as Job;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration;
use Fastmag\Sync\Model\System\Connection\Edi;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Model\System\Connection\Sql\SqlInterface;
use Magento\Framework\App\ResourceConnection;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Integration
{
    /** @var Proxy $proxy */
    protected $proxy;

    /**
     * Save constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Config             $config
     * @param JobRepository      $jobRepository
     * @param Edi                $edi
     * @param Proxy              $proxy
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        Edi $edi,
        Proxy $proxy
    ) {
        parent::__construct($logger, $resourceConnection, $config, $jobRepository, $edi);

        $this->proxy = $proxy;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        if (!$this->isEnabled()) {
            $this->logger->notice('Worker "' . $this->code . '" was called, even though it is disabled');
        } elseif (count($this->jobs) <= 0) {
            $this->logger->notice('Worker "' . $this->code . '" was called, but without jobs to integrate');
        } else {
            /** @var Job $job */
            foreach ($this->jobs as $job) {
                try {
                    $this->processJob($job);
                } catch (JobException $e) {
                    $this->logger->error(
                        '[Job #' . $job->getId() . '] Error on customer\'s address with Magento ID #'
                        . $job->getContentId() . ': ' . $e->getMessage()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());

                    $this->jobRepository->save($job);
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ENABLE);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $jobHydratedData = $job->getHydratedData();

        foreach ($jobHydratedData as $customerId => $hydratedData) {
            if (array_key_exists('addresses', $hydratedData)
                && array_key_exists('shipping_address', $hydratedData['addresses'])
                && $hydratedData['addresses']['shipping_address'] !== null
            ) {
                $shippingAddresses = $hydratedData['addresses']['shipping_address'];

                $this->sendFastmagAddress($customerId, $hydratedData['fastmag_id'], $shippingAddresses);
            }
        }
    }

    /**
     * Send address to Fastmag
     *
     * @param int      $magentoId
     * @param int      $fastmagId
     * @param string[] $addressData
     *
     * @throws JobException
     */
    protected function sendFastmagAddress($magentoId, $fastmagId, $addressData)
    {
        $addressExists = $this->checkFastmagAddress($fastmagId, $addressData['alias']);

        if (!$addressExists) {
            $this->createFastmagAddress($magentoId, $fastmagId, $addressData);
        } else {
            $this->updateFastmagAddress($magentoId, $fastmagId, $addressData);
        }
    }

    /**
     * Check if address exists on Fastmag
     *
     * @param int    $fastmagId
     * @param string $alias
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function checkFastmagAddress($fastmagId, $alias)
    {
        try {
            $sql = 'SELECT AdrLivraison
                FROM clientadresse
		        WHERE AdrLivraison = ' . $this->getSqlConnection()->escape(utf8_decode($alias));

            $rows = $this->getSqlConnection()->get($sql);

            return (bool)count($rows);
        } catch (Exception $e) {
            throw new JobException(__(
                'Error when trying to check customer address on Fastmag. ' .
                'Message: %1. Customer ID: %2. Address alias: %3',
                $e->getMessage(),
                $fastmagId,
                $alias
            ));
        }
    }

    /**
     * Check if the SQL connection to Fastmag is working
     *
     * @return SqlInterface|false
     *
     * @throws NoConnectionException
     */
    protected function getSqlConnection()
    {
        $result = false;

        if ($this->proxy->getConnection()->connect()) {
            $result = $this->proxy->getConnection();
        }

        return $result;
    }

    /**
     * Create address on Fastmag
     *
     * @param int      $magentoId
     * @param int      $fastmagId
     * @param string[] $addressData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function createFastmagAddress($magentoId, $fastmagId, $addressData)
    {
        $ediData = [
            $fastmagId,
            utf8_decode(Text::upper($addressData['alias'])),
            utf8_decode($addressData['lastname'] . ' ' . $addressData['firstname']),
            utf8_decode($addressData['street1']),
            (array_key_exists('street2', $addressData) ? utf8_decode($addressData['street2']) : ''),
            utf8_decode($addressData['postcode']),
            utf8_decode($addressData['city']),
            utf8_decode($addressData['country']),
            '', // Bâtiment
            '', // Digicode 1
            '', // Digicode 2
            '', // Interphone
            '', // Etage
            '', // Porte
            '', // Escalier
            '', // Ascenseur
            utf8_decode($addressData['company']),
            utf8_decode($addressData['phone']),
            '', // Observation
            $addressData['archive']
        ];

        try {
            $this->edi->postInInsertMode(Edi::EDI_LINE_ADDRESS_INSERT, $ediData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to send address\' data of the customer #%1 to Fastmag through EDI: %2. Data sent: %3',
                $magentoId,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Update address on Fastmag
     *
     * @param int      $magentoId
     * @param int      $fastmagId
     * @param string[] $addressData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagAddress($magentoId, $fastmagId, $addressData)
    {
        $this->createFastmagAddress($magentoId, $fastmagId, $addressData);
    }
}
