<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-24
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\CustomerFactory as SyncCustomerFactory;
use Fastmag\Sync\Model\CustomerRepository as SyncCustomerRepository;
use Fastmag\Sync\Model\Jobqueue\ToFastmag as Job;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;
use Fastmag\Sync\Model\Rule\StoresellerRepository;
use Fastmag\Sync\Model\System\Connection\Edi;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Delete
 *
 * Integration class used for deleting customers from Magento to Fastmag
 */
class Delete extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_delete';

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /**
     * Save constructor.
     *
     * @param Logger                 $logger
     * @param ResourceConnection     $resourceConnection
     * @param Config                 $config
     * @param JobRepository          $jobRepository
     * @param Edi                    $edi
     * @param SyncCustomerFactory    $syncCustomerFactory
     * @param SyncCustomerRepository $syncCustomerRepository
     * @param StoresellerRepository  $storesellerRepository
     * @param SearchCriteriaBuilder  $searchCriteriaBuilder
     * @param Json                   $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        Edi $edi,
        SyncCustomerFactory $syncCustomerFactory,
        SyncCustomerRepository $syncCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoresellerRepository $storesellerRepository,
        Json $jsonSerializer
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $config,
            $jobRepository,
            $edi,
            $syncCustomerFactory,
            $syncCustomerRepository,
            $searchCriteriaBuilder,
            $storesellerRepository
        );

        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        if (!$this->isEnabled()) {
            $this->logger->notice('Worker "' . $this->code . '" was called, even though it is disabled');
        } elseif (count($this->jobs) <= 0) {
            $this->logger->notice('Worker "' . $this->code . '" was called, but without jobs to integrate');
        } else {
            /** @var Job $job */
            foreach ($this->jobs as $job) {
                try {
                    $this->processJob($job);
                } catch (JobException $e) {
                    $this->logger->error(
                        '[Job #' . $job->getId() . '] Error on customer with Magento ID #'
                        . $job->getContentId() . ': ' . $e->getMessage()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());

                    $this->jobRepository->save($job);
                }
            }
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $customerId = (int)$job->getContentId();

        try {
            $jobHydratedData = $this->jsonSerializer->unserialize($job->getHydratedData());
        } catch (Exception $e) {
            throw new JobException(__('No hydrated data for the job #%1', $customerId));
        }

        if (!array_key_exists('fastmag_id', $jobHydratedData) || $jobHydratedData['fastmag_id'] === null) {
            $fastmagId = $this->getFastmagIdByEmail($jobHydratedData['email']);

            if ($fastmagId) {
                $jobHydratedData['fastmag_id'] = $fastmagId;

                $this->saveSyncCustomer($customerId, $fastmagId);
            }
        }

        $this->sendInstructionToFastmag($customerId, $jobHydratedData);
        $this->saveSyncCustomer($customerId, $jobHydratedData['fastmag_id'], true);
    }

    /**
     * Send CONSIGNE transaction for the customer deletion to Fastmag
     *
     * @param int   $customerId
     * @param array $hydratedData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendInstructionToFastmag($customerId, $hydratedData)
    {
        $fastmagShop = $this->getFastmagShop($hydratedData['store_id']);

        $ediData = [
            utf8_decode($fastmagShop),
            utf8_decode('SUPPRESSION COMPTE CLIENT'),
            utf8_decode(
                'Bonjour,{CR}{LF}Le client ayant l\'ID Fastmag #' . $hydratedData['fastmag_id']
                . ' doit être supprimé dans Fastmag, conformément à la RGPD.'
            ),
            utf8_decode($this->config->getValue(Config::XML_PATH_CONNECT_EDI_SHOP)),
            '1'
        ];

        try {
            $result = $this->edi->postInInsertMode(Edi::EDI_HEADER_INSTRUCTION_INSERT, $ediData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to send instruction for the deletion of the customer #%1 to Fastmag through EDI: %2',
                $customerId,
                $e->getMessage()
            ));
        }
    }
}
