<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-22
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\CustomerFactory as SyncCustomerFactory;
use Fastmag\Sync\Model\CustomerRepository as SyncCustomerRepository;
use Fastmag\Sync\Model\Jobqueue\ToFastmag as Job;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer\Address\SaveFactory
    as AddressSaveIntegrationWorkerFactory;
use Fastmag\Sync\Model\Rule\StoresellerRepository;
use Fastmag\Sync\Model\System\Connection\Edi;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Serialize\Serializer\Json;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_save';

    /** @var AddressSaveIntegrationWorkerFactory $addressSaveWorkerFactory */
    protected $addressSaveWorkerFactory;

    /** @var Proxy $proxy */
    protected $proxy;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /**
     * Save constructor.
     *
     * @param Logger                              $logger
     * @param ResourceConnection                  $resourceConnection
     * @param Config                              $config
     * @param JobRepository                       $jobRepository
     * @param Edi                                 $edi
     * @param SyncCustomerFactory                 $syncCustomerFactory
     * @param SyncCustomerRepository              $syncCustomerRepository
     * @param SearchCriteriaBuilder               $searchCriteriaBuilder
     * @param AddressSaveIntegrationWorkerFactory $addressSaveWorkerFactory
     * @param StoresellerRepository               $storesellerRepository
     * @param Proxy                               $proxy
     * @param Json                                $jsonSerializer
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        Edi $edi,
        SyncCustomerFactory $syncCustomerFactory,
        SyncCustomerRepository $syncCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        AddressSaveIntegrationWorkerFactory $addressSaveWorkerFactory,
        StoresellerRepository $storesellerRepository,
        Proxy $proxy,
        Json $jsonSerializer
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $config,
            $jobRepository,
            $edi,
            $syncCustomerFactory,
            $syncCustomerRepository,
            $searchCriteriaBuilder,
            $storesellerRepository
        );

        $this->addressSaveWorkerFactory = $addressSaveWorkerFactory;
        $this->proxy = $proxy;
        $this->jsonSerializer = $jsonSerializer;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        if (!$this->isEnabled()) {
            $this->logger->notice('Worker "' . $this->code . '" was called, even though it is disabled');
        } elseif (count($this->jobs) <= 0) {
            $this->logger->notice('Worker "' . $this->code . '" was called, but without jobs to integrate');
        } else {
            /** @var Job $job */
            foreach ($this->jobs as $job) {
                try {
                    $this->processJob($job);
                } catch (JobException $e) {
                    $this->logger->error(
                        '[Job #' . $job->getId() . '] Error on customer with Magento ID #'
                        . $job->getContentId() . ': ' . $e->getMessage()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());

                    $this->jobRepository->save($job);
                }
            }

            $addressSaveWorker = $this->addressSaveWorkerFactory->create();
            $addressSaveWorker->setJobs($this->jobs)->run();
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $customerId = (int)$job->getContentId();

        try {
            $jobHydratedData = $this->jsonSerializer->unserialize($job->getHydratedData());
        } catch (Exception $e) {
            throw new JobException(__('No hydrated data for the job #%1', $customerId));
        }

        if (!array_key_exists('fastmag_id', $jobHydratedData) || $jobHydratedData['fastmag_id'] === null) {
            $fastmagId = $this->getFastmagIdByEmail($jobHydratedData['email']);

            if ($fastmagId) {
                $jobHydratedData['fastmag_id'] = $fastmagId;

                $this->saveSyncCustomer($customerId, $fastmagId);
            }
        }

        $newFastmagId = $this->sendDataToFastmag($customerId, $jobHydratedData);
        if ($newFastmagId !== false) {
            $jobHydratedData['fastmag_id'] = $newFastmagId;

            $this->saveSyncCustomer($customerId, $newFastmagId);
        }
    }

    /**
     * Send customer to Fastmag
     *
     * @param int   $customerId
     * @param array $hydratedData
     *
     * @return int|bool new Fastmag ID if it's an insertion, false otherwise
     *
     * @throws JobException
     */
    protected function sendDataToFastmag($customerId, $hydratedData)
    {
        $result = false;

        if (!array_key_exists('fastmag_id', $hydratedData) || $hydratedData['fastmag_id'] === null) {
            $result = $this->createFastmagCustomer($customerId, $hydratedData);
        } else {
            $this->updateFastmagCustomer($customerId, $hydratedData);
        }

        return $result;
    }

    /**
     * Create customer on Fastmag
     *
     * @param int   $customerId
     * @param array $hydratedData
     *
     * @return int
     *
     * @throws JobException
     */
    protected function createFastmagCustomer($customerId, $hydratedData)
    {
        $fastmagShop = $this->getFastmagShop($hydratedData['store_id']);

        [$dobYear, $dobMonth, $dobDay] = $this->explodeDob($hydratedData['dob']);

        $billingAddress = $this->getBillingAddressFromHydratedData($hydratedData);

        $ediData = [
            utf8_decode($fastmagShop),
            trim(Text::lower($hydratedData['email'])),
            utf8_decode($billingAddress['lastname'] !== '' ?: $hydratedData['lastname']),
            utf8_decode($billingAddress['firstname'] !== '' ?: $hydratedData['firstname']),
            $this->formatGender($hydratedData['gender']),
            utf8_decode($billingAddress['street1']),
            (array_key_exists('street2', $billingAddress) ? utf8_decode($billingAddress['street2']) : ''),
            $billingAddress['postcode'],
            utf8_decode($billingAddress['city']),
            utf8_decode($billingAddress['country']),
            $billingAddress['phone'],
            ($dobDay ?: ''),
            ($dobMonth ?: ''),
            '', // Carte de fidélité
            '', // Remise
            self::DEFAULT_REFERER,
            '', // Observations
            '', // Mot de passe
            utf8_decode($billingAddress['company']),
            '', // Code
            '', // Compte
            '', // Famille
            ($dobYear ?: ''),
            $this->formatGender($hydratedData['gender']),
        ];

        try {
            $result = $this->edi->postInInsertMode(Edi::EDI_HEADER_CUSTOMER_INSERT, $ediData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to send data of the customer #%1 to Fastmag through EDI: %2. Data sent: %3',
                $customerId,
                $e->getMessage(),
                $ediData
            ));
        }

        $this->updateFastmagNewCustomer(
            $customerId,
            $result['id'],
            $hydratedData['newsletter'],
            $hydratedData['created_at']
        );

        return $result['id'];
    }

    /**
     * @param $dob
     *
     * @return string[]
     */
    protected function explodeDob($dob)
    {
        return explode('-', $dob);
    }

    /**
     * Update newley created customer to add some more data
     *
     * @param int    $magentoId
     * @param int    $fastmagId
     * @param bool   $newsletter
     * @param string $createdAt
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagNewCustomer($magentoId, $fastmagId, $newsletter, $createdAt)
    {
        $ediData = [
            'PubParEMail' => $newsletter,
            'DateCreation' => $createdAt
        ];

        try {
            $this->edi->postInUpdateMode(
                Edi::EDI_HEADER_CUSTOMER_UPDATE,
                $fastmagId,
                Edi::EDI_LINE_CUSTOMER_UPDATE,
                $ediData
            );
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to update customer #%1 on Fastmag through EDI: %2. Data sent: %3',
                $magentoId,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Update customer on Fastmag
     *
     * @param int   $magentoId
     * @param array $hydratedData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagCustomer($magentoId, $hydratedData)
    {
        $this->checkFastmagEmail($hydratedData['fastmag_id'], $hydratedData['email']);

        [$dobYear, $dobMonth, $dobDay] = $this->explodeDob($hydratedData['dob']);

        $billingAddress = $this->getBillingAddressFromHydratedData($hydratedData);

        $ediData = [
            'Sexe'           => $this->formatGender($hydratedData['gender']),
            'Nom'            =>
                utf8_decode($billingAddress['lastname'] !== '' ?: $hydratedData['lastname']),
            'Prenom'         =>
                utf8_decode($billingAddress['firstname'] !== '' ?: $hydratedData['firstname']),
            'JourNaissance'  => ($dobDay ?: 'null'),
            'MoisNaissance'  => ($dobMonth ?: 'null'),
            'AnneeNaissance' => ($dobYear ?: 'null'),
            'PubParEMail'    => $hydratedData['newsletter'],
            'DateCreation'   => $hydratedData['created_at'],
            'Societe'        => utf8_decode($billingAddress['company']),
            'Adresse1'       => utf8_decode($billingAddress['street1']),
            'Adresse2'       =>
                (array_key_exists('street2', $billingAddress) ? utf8_decode($billingAddress['street2']) : ''),
            'CodePostal'     => $billingAddress['postcode'],
            'Ville'          => utf8_decode($billingAddress['city']),
            'Pays'           => utf8_decode($billingAddress['country']),
            'Telephone'      => $billingAddress['phone']
        ];

        try {
            $this->edi->postInUpdateMode(
                Edi::EDI_HEADER_CUSTOMER_UPDATE,
                $hydratedData['fastmag_id'],
                Edi::EDI_LINE_CUSTOMER_UPDATE,
                $ediData
            );
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to update customer #%1 on Fastmag through EDI: %2. Data sent: %3',
                $magentoId,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Get billing address from hydrated data
     *
     * @param array $hydratedData
     *
     * @return array
     */
    protected function getBillingAddressFromHydratedData($hydratedData)
    {
        $billingAddress = [
            'lastname'  => '',
            'firstname' => '',
            'street1'   => '',
            'postcode'  => '',
            'city'      => '',
            'country'   => '',
            'phone'     => '',
            'company'   => '',
        ];

        if (array_key_exists('addresses', $hydratedData)
            && array_key_exists('billing_address', $hydratedData['addresses'])
            && $hydratedData['addresses']['billing_address'] !== null
        ) {
            $billingAddress = $hydratedData['billing_address'];
        }

        return $billingAddress;
    }

    /**
     * Check email of the customer on Fastmag, and change it if it's not the same as the one in Magento
     *
     * @param int    $fastmagId
     * @param string $email
     *
     * @return void
     *
     * @throws JobException
     */
    protected function checkFastmagEmail($fastmagId, $email)
    {
        try {
            $sql = 'SELECT Email AS email
                FROM clients 
                WHERE Client = ' . $this->getSqlConnection()->escape($fastmagId);

            $rows = $this->getSqlConnection()->get($sql);

            if ($rows[0]['email'] !== $email) {
                $this->changeCustomerEmail($fastmagId, $email);
            }
        } catch (Exception $e) {
            throw new JobException(__(
                'Error when trying to check customer email on Fastmag. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                $fastmagId
            ));
        }
    }

    /**
     * Change customer email on Fastmag
     *
     * @param string $oldEmail
     * @param string $newEmail
     *
     * @return void
     *
     * @throws JobException
     */
    protected function changeCustomerEmail($oldEmail, $newEmail)
    {
        $ediData = [$oldEmail, $newEmail];

        try {
            $this->edi->postInInsertMode(Edi::EDI_HEADER_CHANGE_EMAIL_INSERT, $ediData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to update the email "%1" on Fastmag through EDI: %2. Data sent: %3',
                $oldEmail,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Format gender for Fastmag
     *
     * @param int $gender
     *
     * @return string
     */
    protected function formatGender($gender)
    {
        $result = 'M';
        $values = [1 => 'M', 2 => 'MMME'];

        if (array_key_exists($gender, $values)) {
            $result = $values[$gender];
        }

        return $result;
    }
}
