<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-17
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Job;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration\Customer\Address;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento\Collection;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Customer
 *
 * Hydration class used for inserting or updating customers from Fastmag to Magento
 */
class Customer extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_customer';

    /** @var Address $addressHydrationWorker */
    protected $addressHydrationWorker;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * Hydration constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Collection         $jobs
     * @param Config             $config
     * @param Proxy              $proxy
     * @param Address            $address
     * @param JobRepository      $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Collection $jobs,
        Config $config,
        Proxy $proxy,
        Address $address,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $jobs, $config, $proxy);

        $this->addressHydrationWorker = $address;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromFastmag();

        if (is_array($results)) {
            foreach ($results as $row) {
                $job = $this->getJob($row['Client']);

                if ($job !== null) {
                    $job->setHydratedData($row);
                }
            }

            $this->addressHydrationWorker->setJobs($this->jobs)->run();

            foreach ($this->jobs as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $customersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT Client, Civilite, Sexe, Nom, Prenom, Societe, Adresse1, Adresse2, CodePostal, Ville,
                    (SELECT CodeIso FROM pays WHERE pays.Pays = clients.Pays) AS PaysIso, Telephone, Portable,
                    email AS Email, JourNaissance, MoisNaissance, AnneeNaissance, Remise, Regroupement, PubParEMail,
                    Actif, Remise, Regroupement, ObsClient, DateCreation
                FROM clients
                WHERE Client IN (' . $this->getConnection()->escape($customersIds) . ')';

            return $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                implode(', ', $customersIds)
            );
        }
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerFastmagId
     *
     * @return Job
     */
    protected function getJob($customerFastmagId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerFastmagId);
    }
}
