<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-06-11
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration\Customer;

use Exception;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Job;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration\Address as DefaultAddress;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Fastmag to Magento
 * Called from Customer hydration worker only
 */
class Address extends DefaultAddress
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_customer_address';

    /** @var array $customersAddresses */
    protected $customersAddresses = [];

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromFastmag();

        if (is_array($results)) {
            foreach ($results as $row) {
                if ($this->shouldAddAddress($row)) {
                    $job = $this->getJob($row['Client']);

                    if ($job !== null) {
                        $customerData = $job->getHydratedData();
                        $customerData['AddressesLivraison'] = $row;
                        $job->setHydratedData($customerData);
                    }

                    if (!array_key_exists($row['Client'], $this->customersAddresses)
                        || $row['Vente'] > $this->customersAddresses[$row['Client']]['last-order']
                    ) {
                        $this->customersAddresses[$row['Client']] = [
                            'last-order' => $row['Vente'],
                            'addresses' => [$row]
                        ];
                    } else {
                        $this->customersAddresses[$row['Client']]['addresses'][] = $row;
                    }
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $customersIds = $this->getCustomersIds();

        try {
            $sql = 'SELECT clientadresse.Client, clientadresse.AdrLivraison, ventes.Vente, Nom, Societe,
                    Adresse1, Adresse2, CodePostal, Ville, Telephone,
                    (SELECT CodeIso FROM pays WHERE pays.Pays = clientadresse.Pays) as PaysIso
                FROM clientadresse LEFT OUTER JOIN ventes
                    ON clientadresse.AdrLivraison = ventes.AdrLivraison AND clientadresse.Client = ventes.Client
                WHERE Archiver = 0 AND TRIM(clientadresse.AdrLivraison) <> \'\'
                    AND clientadresse.Client = ' . $this->getConnection()->escape($customersIds) . '
                ORDER BY clientadresse.Client ASC, ventes.Vente DESC';

            return $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException(
                'Error when hydrating customers addresses. Message: ' . $e->getMessage()
                . '. Customers IDs: ' . implode(', ', $customersIds)
            );
        }
    }

    /**
     * Returns the list of the customers IDs, if the jobs is linked to customers.
     *
     * @return int[]
     */
    protected function getCustomersIds()
    {
        return $this->jobs->getColumnValues(Job::CONTENT_ID);
    }

    /**
     * Defines if the address data given in parameters should be added to the customer addresses
     *
     * @param array $row
     *
     * @return bool
     */
    protected function shouldAddAddress($row)
    {
        $result = true;

        if (array_key_exists($row['Client'], $this->customersAddresses)
            && $this->customersAddresses[$row['Client']]['last-order'] !== null
            && $row['Vente'] < $this->customersAddresses[$row['Client']]['last-order']
        ) {
            $result = false;
        }

        return $result;
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerFastmagId
     *
     * @return Job
     */
    protected function getJob($customerFastmagId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerFastmagId);
    }
}
