<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-26
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Job;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository as JobRepository;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration;
use Fastmag\Sync\Model\Rule\Storeseller;
use Fastmag\Sync\Model\Rule\StoresellerRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento\Collection;
use Fastmag\Sync\Model\System\Connection\Proxy;
use InvalidArgumentException;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Api\StoreRepositoryInterface;

/**
 * Class Product
 *
 * Hydration class used for inserting or updating products from Fastmag to Magento
 */
class Product extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product';

    /** @var ProductFactory $productFactory */
    protected $productFactory;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var StoresellerRepository $storesellerRepository */
    protected $storesellerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var StoreRepositoryInterface $storeRepository */
    protected $storeRepository;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var Storeseller $currentRule */
    protected $currentRule;

    /** @var array $colorsI18n*/
    protected $colorsI18n;

    /** @var array $sizesI18n*/
    protected $sizesI18n;

    /** @var array $referenceStocks */
    protected $referenceStocks;

    /**
     * Product constructor
     *
     * @param Logger                   $logger
     * @param ResourceConnection       $resourceConnection
     * @param Collection               $jobs
     * @param Config                   $config
     * @param Proxy                    $proxy
     * @param ProductFactory           $productFactory
     * @param Json                     $jsonSerializer
     * @param StoresellerRepository    $storesellerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param StoreRepositoryInterface $storeRepository
     * @param JobRepository            $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Collection $jobs,
        Config $config,
        Proxy $proxy,
        ProductFactory $productFactory,
        Json $jsonSerializer,
        StoresellerRepository $storesellerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreRepositoryInterface $storeRepository,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $jobs, $config, $proxy);

        $this->productFactory = $productFactory;
        $this->jsonSerializer = $jsonSerializer;
        $this->storesellerRepository = $storesellerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeRepository = $storeRepository;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     * @throws JobException
     */
    public function run()
    {
        $this->getGlobalData();

        /** @var Job $job */
        foreach ($this->jobs as $job) {
            $this->currentJob = $job;
            $hydratedData = $this->getDataFromFastmag();

            $job->setHydratedData($hydratedData);

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $e) {
                $this->logger->critical(
                    'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                    . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                );
            }
        }
    }

    /**
     * Hydrate the product of the current job
     *
     * @return array
     *
     * @throws JobException
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();
        [$fastmagParentRef, $fastmagColorRef] = $this->getRealFastmagRef();

        if ($this->isPack($fastmagParentRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $hydratedData = [
            'colors_18n' => $this->colorsI18n,
            'sizes_18n'  => $this->sizesI18n,
            'parent_ref' => $fastmagParentRef,
            'color_ref'  => $fastmagColorRef,
            'is_color'   => ($fastmagColorRef !== ''),
            'magento_id' => $this->getMagentoProductId($fastmagRef)
        ];

        $hydratedData = array_merge($hydratedData, $this->getBasicData($fastmagParentRef));

        if ($this->productMustBeSynced($hydratedData)) {
            $storesList = $this->storeRepository->getList();

            foreach ($storesList as $store) {
                try {
                    $this->currentRule = $this->storesellerRepository->getByStoreId($store->getId());
                } catch (Exception $e) {
                    continue;
                }

                $hydratedDataStore['vat_rate'] = $this->getFastmagShopTaxRate($this->currentRule->getFastmagShop());

                $fastmagStaticData = $this->getStaticData($fastmagParentRef, $fastmagColorRef);
                $hydratedDataStore['children'] = $fastmagStaticData;

                $fastmagI18nData = $this->getI18nData($fastmagParentRef);
                $hydratedDataStore['i18n'] = $fastmagI18nData;

                $hydratedData[$store->getCode()] = $hydratedDataStore;
            }
        }

        return $hydratedData;
    }

    /**
     * Get global data (independent of the jobs)
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getGlobalData()
    {
        $this->getColorsI18n();
        $this->getSizesI18n();
    }

    /**
     * Get colors labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getColorsI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Couleur AS color, Traduction AS translation FROM couleurslangues';

            $this->colorsI18n = $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }
    }

    /**
     * Get sizes labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getSizesI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Taille AS size, Traduction AS translation FROM tailleslangues';

            $this->sizesI18n = $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }
    }

    /**
     * Get the parent and child refs from job data
     *
     * @return array
     */
    protected function getRealFastmagRef()
    {
        $content = $this->jsonSerializer->unserialize($this->currentJob->getContent());

        $refs = [$this->currentJob->getContentId(), ''];
        if ($content['type_reference'] === 'color') {
            $refs = explode('-', $this->currentJob->getContentId());
        }

        return $refs;
    }

    /**
     * Check if product is a pack
     *
     * @param string $fastmagRef
     *
     * @return bool
     */
    protected function isPack($fastmagRef)
    {
        $result = false;

        if (strpos($fastmagRef, 'PACK_') !== false) {
            $result = true;
        }

        return $result;
    }

    /**
     * Get Magento product ID for Fastmag ref given
     *
     * @param string $fastmagRef
     *
     * @return int
     */
    protected function getMagentoProductId($fastmagRef)
    {
        $product = $this->productFactory->create();

        return $product->getIdBySku($fastmagRef);
    }

    /**
     * Get product basic data on Fastmag (VisibleWeb, Actif, creation date, inventory data, family)
     *
     * @param string $fastmagParentRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getBasicData($fastmagParentRef)
    {
        try {
            $stockListCondition = '';

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $stockList = $this->getReferenceStocks();

                if (!is_array($stockList) || count($stockList) === 0) {
                    throw new ProcessException(
                        'No store/seller rules is configured, please create some rules,'
                        . ' or disable "Only import declinations received by sellers" config field.'
                    );
                }

                $stockListCondition = ' AND s.CodeMag IN (' . $this->getConnection()->escape($stockList) . ')';
            }

            $sql = 'SELECT pf.VisibleWeb, pf.Actif, pf.DateCreation, COUNT(s.ID) AS Stocks,
                    GROUP_CONCAT(DISTINCT IF(Stock >= 1, s.CodeMag, NULL)) AS CodeMagList, pf.Famille, f.TauxTVA
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                    AND s.AR = 1 ' . $stockListCondition . '
                GROUP BY pf.BarCode';

            $rows = $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        $row = reset($rows);

        return [
            'visible_web' => $row['VisibleWeb'] ?? 0,
            'active' => $row['Actif'] ?? 0,
            'creation_date' => $row['DateCreation'] ?? '',
            'stock_level' => $row['Stocks'] ?? 0,
            'stocks_list' => $row['CodeMagList'] ? explode(',', $row['CodeMagList']) : '',
            'family' => $row['Famille'] ?? '',
            'vat_rate' => $row['TauxTVA'] ?? ''
        ];
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param array $hydratedData
     *
     * @return bool
     */
    protected function productMustBeSynced($hydratedData)
    {
        $result = ($hydratedData['active'] && $hydratedData['visible_web'])
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyStoreseller = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER);
        if ($hydratedData['stock_level'] === 0 && $onlyStoreseller) {
            $result = false;
        }

        if ($hydratedData['magento_id']) {
            $result = true;
        }

        $parentPerColor = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_PARENT_PER_COLOR);
        $genericProduct = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_GENERIC_PRODUCTS);
        if (!$hydratedData['is_color'] && !$hydratedData['magento_id'] && $parentPerColor && !$genericProduct) {
            $result = false;
        }

        return $result;
    }

    /**
     * Set all stock references used for synchronization
     *
     * @return array
     */
    protected function getReferenceStocks()
    {
        if ($this->referenceStocks === null) {
            $searchCriteria = $this->searchCriteriaBuilder->create();
            $storesellerRuleList = $this->storesellerRepository->getList($searchCriteria)->getItems();
            $stockReferences = [];

            /** @var Storeseller $rule */
            foreach ($storesellerRuleList as $rule) {
                if (!in_array($rule->getFastmagShop(), $stockReferences, true)) {
                    $stockReferences[] = $rule->getFastmagShop();
                }

                $referencesStock = explode('|', $rule->getReferenceStock());
                foreach ($referencesStock as $stock) {
                    if (!in_array($stock, $stockReferences, true)) {
                        $stockReferences[] = $stock;
                    }
                }
            }

            $this->referenceStocks = $stockReferences;
        }

        return $this->referenceStocks;
    }

    /**
     * Get tax rate for the product
     *
     * @param string $fastmagShop
     * @param float  $defaultValue
     *
     * @return float
     *
     * @throws ProcessException
     */
    protected function getFastmagShopTaxRate($fastmagShop, $defaultValue = 20.0)
    {
        $result = $defaultValue;

        try {
            $sql = 'SELECT TVA FROM mag WHERE Code = ' . $this->getConnection()->escape($fastmagShop);

            $rows = $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $result = $row['TVA'];
            }
        }

        return $result;
    }

    /**
     * Get static data for the product from Fastmag
     *
     * @param string      $fastmagParentRef
     * @param string      $fastmagColorRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getStaticData($fastmagParentRef, $fastmagColorRef)
    {
        $result = [];

        try {
            $rateCode = $this->currentRule->getRateCode();
            $fastmagShop = $this->currentRule->getFastmagShop();

            $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagParentRef, $fastmagShop, $rateCode);
            $combinationPriceSubquery = $this->getCombinationPriceSubquery($fastmagParentRef, $rateCode);
            $priceRateSubquery = $this->getPriceRateSubquery($fastmagParentRef, $rateCode);
            $salesPriceSubquery = $this->getSalesPriceSubquery($fastmagParentRef, $rateCode);
            $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
            $buyingPriceSubquery = $this->getBuyingPriceSubquery($fastmagParentRef);

            $onlyStoresellerCombination = ' AND (no_web < 1 OR no_web IS NULL)';
            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $onlyStoresellerCombination .= ' AND enable_stock = 1 ';
            }

            $colorCondition = '';
            if ($fastmagColorRef !== '') {
                $colorCondition = ' AND stock.Couleur = ' . $this->getConnection()->escape($fastmagColorRef);
            }

            $stockCondition = '';
            $referenceStocks = $this->getReferenceStocks();
            if ($referenceStocks !== null && count($referenceStocks) !== 0) {
                $stockCondition = ' AND CodeMag IN (' . $this->getConnection()->escape($referenceStocks) . ')';
            }

            $sql = 'SELECT
                    (
                        SELECT JSON_OBJECT(
                            \'family\', Famille,
                            \'subfamily\', SsFamille,
                            \'season\', Saison,
                            \'section\', Rayon,
                            \'brand\', Marque
                        )
                        FROM produitsfiches AS pf
                        WHERE pf.BarCode = stock.BarCode
                    ) AS rules_data,
                    (
                        SELECT JSON_OBJECT(
                            \'fastmag_product_id\', Produit,
                            \'ean13\', RIGHT(GenCod, 13),
                            \'supplier_ref\', RefFournisseur,
                            \'weight\', PoidsTC
                        )
                        FROM produits AS p
                        WHERE p.BarCode = stock.BarCode AND p.Taille = stock.Taille AND p.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS combination_data,
                    (
                        SELECT ProduitActif
                        FROM produits AS p2
                        WHERE p2.BarCode = stock.BarCode AND p2.Taille = stock.Taille AND p2.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS enable,
                    (
                        SELECT 1
                        FROM stock AS s2
                        WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                            ' . $stockCondition . '
                        LIMIT 1
                    ) AS enable_stock,
                    (
                        SELECT IFNULL(ValeurInteger, -1) AS no_web
                        FROM produits AS p3 LEFT JOIN complementsr AS c
                            ON p3.Produit = c.Clef AND c.Champ = \'NO_WEB\' AND c.Nature = \'PRODUITSTC\'
                        WHERE p3.BarCode = stock.BarCode AND p3.Taille = stock.Taille AND p3.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS no_web,
                    (' . $standardPriceSubquery . ') AS standard_price,
                    (' . $combinationPriceSubquery . ') AS combination_price,
                    (' . $priceRateSubquery . ') AS price_rate,
                    (' . $salesPriceSubquery . ') AS sales_price,
                    (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                    (' . $buyingPriceSubquery . ') AS buying_price,
                    stock.Taille AS size,
                    stock.Couleur AS color,
                    (
                        SELECT Descriptif
                        FROM produitscouleurs AS pc
                        WHERE pc.Barcode = stock.BarCode AND pc.Couleur = stock.Couleur
                    ) AS alternative_color,
                    SUM(Stock) AS total_stock,
                    Poids AS weight,
                    stock.AR AS in_stock
                FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
                WHERE stock.AR = 1 AND stock.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                    ' . $colorCondition . '
                GROUP BY stock.Barcode, stock.Taille, stock.Couleur
                HAVING enable = 1 ' . $onlyStoresellerCombination . '
                ORDER BY stock.Couleur, stock.Taille';

            $rows = $this->getConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        foreach ($rows as $row) {
            $row = $this->extractJsonData($row);

            if (!array_key_exists($row['color'], $result)) {
                $result[$row['color']] = [$row['size'] => $row];
            } else {
                $result[$row['color']][$row['size']] = $row;
            }
        }

        return $result;
    }

    /**
     * Returns subquery for stock price (standard price)
     *
     * @param string $fastmagParentRef
     * @param string $fastmagShop
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getStandardPriceSubquery($fastmagParentRef, $fastmagShop, $priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT stpr.PrixVente
                    FROM stock AS stpr
                    WHERE stpr.CodeMag = ' . $this->getConnection()->escape($fastmagShop) . '
                        AND stpr.AR = 1
                        AND stpr.Taille = stock.Taille
                        AND stpr.Couleur = stock.Couleur
                        AND stpr.PrixVente > 0
                        AND stpr.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                    ORDER BY stpr.Date, stpr.ID
                    LIMIT 1';
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for combination price
     *
     * @param string $fastmagParentRef
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getCombinationPriceSubquery($fastmagParentRef, $priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT p.PrixVente
                    FROM produits AS p
                    WHERE p.PrixVente > 0
                        AND p.Taille = stock.Taille
                        AND p.Couleur = stock.Couleur
                        AND p.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true);
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for combination price
     *
     * @param string $fastmagParentRef
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getPriceRateSubquery($fastmagParentRef, $priceRateCode)
    {
        try {
            // Specific tarif price
            if ($priceRateCode !== '') {
                if ($priceRateCode === 'prix_indicatif') {
                    $result = 'pf.PrixVente';
                } else {
                    $result = 'SELECT pt.Prix
                        FROM produitstarifs AS pt
                        WHERE pt.Tarif = ' . $this->getConnection()->escape($priceRateCode) . '
                            AND pt.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                            AND pt.Taille = stock.Taille AND pt.Prix > 0 AND pt.Couleur = stock.Couleur
                        ORDER BY pt.Prix desc
                        LIMIT 1';
                }
            } else {
                $result = '0';
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for sales price
     *
     * @param string $fastmagParentRef
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getSalesPriceSubquery($fastmagParentRef, $fastmagShop)
    {
        try {
            $result = 'SELECT JSON_OBJECT(
                        \'sales_price\', Prix,
                        \'discount\', Remise,
                        \'begin_at\', DateDebut,
                        \'end_at\', DateFin,
                        \'reason\', Motif
                    )
                FROM prixremise
                WHERE CodeMag = ' . $this->getConnection()->escape($fastmagShop) . '
                    AND BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                    AND DateDebut <= date(NOW()) AND DateFin >= date(NOW())
                ORDER BY DateDebut DESC, DateFin DESC, Remise, Prix
                LIMIT 1';
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for size and color sales price
     *
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getSalesPriceChildrenSubquery($fastmagShop)
    {
        try {
            $result = 'SELECT JSON_OBJECT(
                        \'sales_price\', Prix,
                        \'discount\', Remise,
                        \'begin_at\', DateDebut,
                        \'end_at\', DateFin,
                        \'reason\', Motif
                    )
                FROM prixremisetc AS ptc
                WHERE ptc.CodeMag = ' . $this->getConnection()->escape($fastmagShop) . '
                    AND ptc.BarCode = stock.BarCode AND ptc.Taille = stock.Taille AND ptc.Couleur = stock.Couleur
                    AND (ptc.Remise > 0 OR ptc.Prix > 0) AND DateDebut <= date(NOW()) AND ptc.DateFin >= date(NOW())
                ORDER BY ptc.DateDebut DESC, ptc.DateFin DESC, ptc.Remise, ptc.Prix
                LIMIT 1';
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for buying price
     *
     * @param string $fastmagParentRef
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getBuyingPriceSubquery($fastmagParentRef)
    {
        try {
            $result = 'SELECT prod.PrixAchat
                FROM produits AS prod
                WHERE prod.Taille = stock.Taille AND prod.Couleur = stock.Couleur
                    AND prod.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                LIMIT 1';
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Extract JSON data and merge them to the row given
     *
     * @param array $row
     *
     * @return array
     */
    protected function extractJsonData($row)
    {
        try {
            $rulesData = $this->jsonSerializer->unserialize($row['rules_data']);
            $row = array_merge($row, $rulesData);
            unset($row['rules_data']);
        } catch (InvalidArgumentException $e) {
            $this->logger->warning(
                '[Job #' . $this->currentJob->getId() . '] Error when unserializing rules_data from Fastmag: '
                . $row['rules_data']
            );
        }

        try {
            $combinationData = $this->jsonSerializer->unserialize($row['combination_data']);
            $row = array_merge($row, $combinationData);
            unset($row['combination_data']);
        } catch (InvalidArgumentException $e) {
            $this->logger->warning(
                '[Job #' . $this->currentJob->getId()
                . '] Error when unserializing combination_data from Fastmag: ' . $row['combination_data']
            );
        }

        return $row;
    }

    /**
     * Get minimal price (indicative value) from Fastmag, if price can not be synced by other means
     *
     * @param string $fastmagParentRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getMinimalPrice($fastmagParentRef)
    {
        try {
            $sql = 'SELECT PrixVente AS PrixVenteIndicatif
                FROM produits
                WHERE BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                ORDER BY PrixVente
                LIMIT 1';

            $rows = $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return reset($rows);
    }

    /**
     * Get all i18nized data from Fastmag
     *
     * @todo add store parameter to actually retrieve i18nized data
     *
     * @param string $fastmagParentRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getI18nData($fastmagParentRef)
    {
        $additionalAttributes = $this->getAdditionalAttributes();

        try {
            $sql = 'SELECT \'FR\' AS language, Designation AS designation, Designation2 AS designation2,
                    Marque AS brand, Fournisseur AS supplier, Lavage AS washing, Chlore AS chlorine,
                    Repassage AS ironing, Pressing AS dry_cleaning, Sechage AS drying, AquaNettoyage AS water_cleaning,
                    Descriptif AS description, DateCreation AS created_at' . $additionalAttributes . '
                FROM produitsfiches
                WHERE BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true);

            $rows = $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        $result['FR'] = reset($rows);

        return $result;
    }

    /**
     * Returns SQL to get data for additionnal attributes, given config fields
     *
     * @return string
     */
    protected function getAdditionalAttributes()
    {
        $additionalAttributes = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $additionalAttributes[] = 'Famille AS family';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $additionalAttributes[] = 'SsFamille AS subfamily';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $additionalAttributes[] = 'Rayon AS section';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $additionalAttributes[] = 'Saison AS season';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $additionalAttributes[] = 'Modele AS model';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $additionalAttributes[] = 'Matiere AS material';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $additionalAttributes[] = 'Theme AS theme';
        }

        return ', ' . implode(', ', $additionalAttributes);
    }
}
