<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-07-09
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToMagento\Integration\Customer;

use DateTime;
use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Customer as SyncCustomer;
use Fastmag\Sync\Model\CustomerFactory as SyncCustomerFactory;
use Fastmag\Sync\Model\CustomerRepository as SyncCustomerRepository;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Job;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository as JobRepository;
use Fastmag\Sync\Model\Process\Entity\ToMagento\Customer as CustomerEntity;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Integration\Customer as CustomerIntegrationWorker;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Integration\Customer\Address\SaveFactory
    as AddressSaveIntegrationWorkerFactory;
use Magento\Customer\Api\AccountManagementInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Customer\Api\Data\CustomerInterfaceFactory;
use Magento\Customer\Api\GroupManagementInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\Pdo\Mysql;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Newsletter\Model\SubscriberFactory;
use Magento\Store\Api\WebsiteRepositoryInterface;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_customer_create and tomagento_integration_customer_update jobs
 */
class Save extends CustomerIntegrationWorker
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_customer_save';

    /** @var SyncCustomerFactory $syncCustomerFactory */
    protected $syncCustomerFactory;

    /** @var SyncCustomerRepository $syncCustomerRepository */
    protected $syncCustomerRepository;

    /** @var CustomerRepositoryInterface $customerRepository */
    protected $customerRepository;

    /** @var CustomerInterfaceFactory $customerFactory */
    protected $customerFactory;

    /** @var WebsiteRepositoryInterface $websiteRepository */
    protected $websiteRepository;

    /** @var AccountManagementInterface $accountManagement */
    protected $accountManagement;

    /** @var AddressSaveIntegrationWorkerFactory $addressSaveWorkerFactory */
    protected $addressSaveWorkerFactory;

    /** @var GroupManagementInterface $groupManagement */
    protected $groupManagement;

    /** @var SubscriberFactory $subscriberFactory */
    protected $subscriberFactory;

    /**
     * Save constructor
     *
     * @param Logger                              $logger
     * @param ResourceConnection                  $resourceConnection
     * @param Config                              $config
     * @param JobRepository                       $jobRepository
     * @param SyncCustomerFactory                 $syncCustomerFactory
     * @param SyncCustomerRepository              $syncCustomerRepository
     * @param CustomerRepositoryInterface         $customerRepository
     * @param CustomerInterfaceFactory            $customerFactory
     * @param WebsiteRepositoryInterface          $websiteRepository
     * @param AccountManagementInterface          $accountManagement
     * @param AddressSaveIntegrationWorkerFactory $addressSaveWorkerFactory
     * @param GroupManagementInterface            $groupManagement
     * @param SubscriberFactory                   $subscriberFactory
     *
     * @SuppressWarnings(PHPMD.ExcessiveParameterList)
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        SyncCustomerFactory $syncCustomerFactory,
        SyncCustomerRepository $syncCustomerRepository,
        CustomerRepositoryInterface $customerRepository,
        CustomerInterfaceFactory $customerFactory,
        WebsiteRepositoryInterface $websiteRepository,
        AccountManagementInterface $accountManagement,
        AddressSaveIntegrationWorkerFactory $addressSaveWorkerFactory,
        GroupManagementInterface $groupManagement,
        SubscriberFactory $subscriberFactory
    ) {
        parent::__construct($logger, $resourceConnection, $config, $jobRepository);

        $this->syncCustomerFactory = $syncCustomerFactory;
        $this->syncCustomerRepository = $syncCustomerRepository;
        $this->customerRepository = $customerRepository;
        $this->customerFactory = $customerFactory;
        $this->websiteRepository = $websiteRepository;
        $this->accountManagement = $accountManagement;
        $this->addressSaveWorkerFactory = $addressSaveWorkerFactory;
        $this->groupManagement = $groupManagement;
        $this->subscriberFactory = $subscriberFactory;
        $this->syncCustomerFactory = $syncCustomerFactory;
        $this->syncCustomerRepository = $syncCustomerRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        if (!$this->isEnabled()) {
            $this->logger->notice('Worker "' . $this->code . '" was called, even though it is disabled');
        } elseif (count($this->jobs) <= 0) {
            $this->logger->notice('Worker "' . $this->code . '" was called, but without jobs to integrate');
        } else {
            /** @var Job $job */
            foreach ($this->jobs as $job) {
                try {
                    $this->processJob($job);
                } catch (JobException $e) {
                    $this->logger->error(
                        '[Job #' . $job->getId() . '] Error on customer with Fastmag ID #'
                        . $this->getJobEntityId($job) . ': ' . $e->getMessage()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());

                    $this->jobRepository->save($job);
                }
            }

            $addressSaveWorker = $this->addressSaveWorkerFactory->create();
            $addressSaveWorker->setJobs($this->jobs)->run();
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        if (!$entity->getEmailAddress()) {
            throw new JobException(__(
                'Email is invalid: %1',
                $entity->getOrigData('email_address')
            ));
        }

        if (!$entity->getIsActive() && !$this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_INACTIVE_CUSTOMERS)) {
            throw new JobException(__('The customer is not active on Fastmag'));
        }

        $customer = $this->checkIfCustomerExists($entity);
        if (!$customer) {
            $customer = $this->customerFactory->create();
        }

        $this->saveEntity($customer, $entity);
    }

    /**
     * Save customer with entity attributes
     *
     * @todo : find a way to use deeper parts of the core code to decrease memory usage and processing time
     *
     * @param CustomerInterface $customer
     * @param CustomerEntity    $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveEntity($customer, $entity)
    {
        $this->setCustomerData($customer, $entity);

        if (!$customer->getId() && $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ACCOUNT_EMAIL)) {
            try {
                // If there is an error on the URL of the mail sent, please check README.md
                $customer = $this->accountManagement->createAccount($customer, null);
            } catch (LocalizedException $e) {
                throw new JobException(__(
                    'Unable to sync the customer: %1',
                    $e->getMessage()
                ));
            }
        } else {
            try {
                $this->customerRepository->save($customer);
            } catch (LocalizedException $e) {
                throw new JobException(__(
                    'Unable to sync the customer: %1',
                    $e->getMessage()
                ));
            }
        }

        $entity->setMagentoId($customer->getId())
            ->setMagentoDefaultBilling($customer->getDefaultBilling())
            ->setMagentoDefaultShipping($customer->getDefaultShipping());

        $this->saveNewsletterPreferences($customer, $entity);

        $this->addToSyncedCustomers($customer, $entity);
    }

    /**
     * Check if a customer in Magento DB has the ID or the email address given in parameter
     * and returns it if it exists
     *
     * @param CustomerEntity $entity
     *
     * @return CustomerInterface|false
     */
    protected function checkIfCustomerExists($entity)
    {
        try {
            $fastmagCustomer = $this->syncCustomerRepository->getByFastmagId($entity->getFastmagId());

            if ($fastmagCustomer->getId()) {
                $customer = $this->customerRepository->getById($fastmagCustomer->getMagentoCustomerId());
            } else {
                $customer = $this->customerRepository->getById($entity->getMagentoId());

                if (!$customer->getId()) {
                    $customer = $this->customerRepository->get($entity->getEmailAddress());
                }
            }
        } catch (LocalizedException $e) {
            $customer = false;
        }

        return $customer;
    }

    /**
     * Get job's entity ID
     *
     * @param Job $job
     *
     * @return int
     */
    protected function getJobEntityId($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        return $entity->getFastmagId();
    }

    /**
     * Set entity data into customer model
     *
     * @param CustomerInterface $customer
     * @param CustomerEntity    $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function setCustomerData($customer, $entity)
    {
        /* @todo : allow (with config) to import customer to a different website */
        $customer->setWebsiteId($this->websiteRepository->getDefault()->getId());

        $customer->setEmail($entity->getEmailAddress())
            ->setGender($entity->getGender())
            ->setPrefix($entity->getPrefix())
            ->setFirstname($entity->getFirstname())
            ->setLastname($entity->getLastname())
            ->setDob($entity->getBirthday())
            ->setCreatedAt($entity->getCreatedAt());

        $customer->setConfirmation(null);

        $this->setGroupCustomer($customer, $entity);
    }

    /**
     * Try to set customer group to customer
     *
     * @param CustomerInterface $customer
     * @param CustomerEntity    $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function setGroupCustomer($customer, $entity)
    {
        try {
            $groupId = $entity->getCustomerGroupId();
            if (!$groupId) {
                $groupId = $this->groupManagement->getDefaultGroup()->getId();
            }

            $customer->setGroupId($groupId);
        } catch (JobException $e) {
            throw $e;
        } catch (LocalizedException $e) {
            throw new JobException(__(
                'Unable to set group to the customer: %1',
                $e->getMessage()
            ));
        }
    }

    /**
     * Try to set customer group to customer
     *
     * @param CustomerInterface $customer
     * @param CustomerEntity    $entity
     *
     * @return void
     */
    protected function saveNewsletterPreferences($customer, $entity)
    {
        if ($entity->getSubscribeNewsletter()) {
            $this->subscriberFactory->create()->subscribeCustomerById($customer->getId());
        } else {
            $this->subscriberFactory->create()->unsubscribeCustomerById($customer->getId());
        }
    }

    /**
     * Add a line in synced customers table
     *
     * @param CustomerInterface $customer
     * @param CustomerEntity    $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function addToSyncedCustomers($customer, $entity)
    {
        $syncCustomer = $this->getSyncedCustomer($customer, $entity);
        $syncCustomer->setMagentoCustomerId($customer->getId())
            ->setFastmagCustomerId($entity->getFastmagId());

        try {
            $now = new DateTime();
            $now = $now->format(Mysql::DATETIME_FORMAT);
        } catch (Exception $e) {
            $now = date(Mysql::DATETIME_FORMAT);
        }
        $syncCustomer->setProcessedAt($now);

        try {
            $this->syncCustomerRepository->save($syncCustomer);
        } catch (CouldNotSaveException $e) {
            throw new JobException(__($e->getMessage()));
        }
    }

    /**
     * Get corresponding line on synced customers table
     *
     * @param CustomerInterface $customer
     * @param CustomerEntity    $entity
     *
     * @return SyncCustomer
     *
     * @throws JobException
     */
    protected function getSyncedCustomer($customer, $entity)
    {
        try {
            $syncCustomer = $this->syncCustomerRepository->getByMagentoId($customer->getId());
        } catch (NoSuchEntityException $e) {
            $syncCustomer = false;
        } catch (LocalizedException $e) {
            throw new JobException(__($e->getMessage()));
        }

        if (!$syncCustomer) {
            try {
                $syncCustomer = $this->syncCustomerRepository->getByFastmagId($entity->getFastmagId());
            } catch (NoSuchEntityException $e) {
                $syncCustomer = false;
            } catch (LocalizedException $e) {
                throw new JobException(__($e->getMessage()));
            }

            if ($syncCustomer && $syncCustomer->getId()) {
                throw new JobException(__(
                    'Customer with Fastmag ID "%1" already synced with customer #%2, but just saved as customer #%3',
                    $entity->getFastmagId(),
                    $syncCustomer->getId(),
                    $customer->getId()
                ));
            }
        }

        if (!$syncCustomer) {
            $syncCustomer = $this->syncCustomerFactory->create();
        }

        return $syncCustomer;
    }
}
