<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-06
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection\Edi;

use Fastmag\Sync\Model\Config\Provider\Edi as EdiConfig;

/**
 * Class Config
 *
 * Class providing CURL params by EDI config
 */
class Config
{
    /** @var string $host */
    protected $host;

    /** @var string $port */
    protected $port;

    /** @var string $chain */
    protected $chain;

    /** @var string $shop */
    protected $shop;

    /** @var string $user */
    protected $user;

    /** @var string $password */
    protected $password;

    /** @var bool $ssl */
    protected $ssl;

    /** @var EdiConfig $configProvider */
    protected $configProvider;

    /**
     * Config constructor
     *
     * @param EdiConfig  $configProvider
     */
    public function __construct(EdiConfig $configProvider)
    {
        $this->configProvider = $configProvider;

        $this->setConfigFromProvider();
    }

    /**
     * Get host
     *
     * @return string|null
     */
    public function getHost()
    {
        return $this->host;
    }

    /**
     * Get port
     *
     * @return string|null
     */
    public function getPort()
    {
        return $this->port;
    }

    /**
     * Get chain
     *
     * @return string|null
     */
    public function getChain()
    {
        return $this->chain;
    }

    /**
     * Get shop
     *
     * @return string|null
     */
    public function getShop()
    {
        return $this->shop;
    }

    /**
     * Get user
     *
     * @return string|null
     */
    public function getUser()
    {
        return $this->user;
    }

    /**
     * Get password
     *
     * @return string|null
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * Get SSL
     *
     * @return string|null
     */
    public function getSsl()
    {
        return $this->ssl;
    }

    /**
     * Set host
     *
     * @param string $value
     *
     * @return self
     */
    public function setHost($value)
    {
        $this->host = $value;

        return $this;
    }

    /**
     * Set port
     *
     * @param string $value
     *
     * @return self
     */
    public function setPort($value)
    {
        $this->port = $value;

        return $this;
    }

    /**
     * Set chain
     *
     * @param string $value
     *
     * @return self
     */
    public function setChain($value)
    {
        $this->chain = $value;

        return $this;
    }

    /**
     * Set shop
     *
     * @param string $value
     *
     * @return self
     */
    public function setShop($value)
    {
        $this->shop = $value;

        return $this;
    }

    /**
     * Set user
     *
     * @param string $value
     *
     * @return self
     */
    public function setUser($value)
    {
        $this->user = $value;

        return $this;
    }

    /**
     * Set password
     *
     * @param string $value
     *
     * @return self
     */
    public function setPassword($value)
    {
        $this->password = $value;

        return $this;
    }

    /**
     * Set SSL
     *
     * @param bool $value
     *
     * @return self
     */
    public function setSsl($value)
    {
        $this->ssl = $value;

        return $this;
    }

    /**
     * Transfer config from provider to internal config
     *
     * @return void
     */
    protected function setConfigFromProvider()
    {
        if ($this->configProvider !== null) {
            $this->setHost($this->configProvider->getHost());
            $this->setPort($this->configProvider->getPort());
            $this->setChain($this->configProvider->getChain());
            $this->setShop($this->configProvider->getShop());
            $this->setUser($this->configProvider->getUser());
            $this->setPassword($this->configProvider->getPassword());
            $this->setSsl($this->configProvider->getSsl());
        }
    }

    /**
     * Transfer config from param
     *
     * @param array $config
     *
     * @return void
     */
    public function setConfigFromParam($config)
    {
        if (is_array($config) && $this->validateConfig($config)) {
            $this->setHost($config['host']);
            $this->setPort($config['port']);
            $this->setChain($config['chain']);
            $this->setShop($config['shop']);
            $this->setUser($config['user']);
            $this->setPassword($config['password']);
            $this->setSsl($config['ssl']);
        }
    }

    /**
     * Get formatted config
     *
     * @return array
     */
    public function getFormattedConfig()
    {
        return [
            'host'     => $this->getHost(),
            'port'     => $this->getPort(),
            'chain'    => $this->getChain(),
            'shop'     => $this->getShop(),
            'user'     => $this->getUser(),
            'password' => $this->getPassword(),
            'ssl'      => $this->getSsl(),
        ];
    }

    /**
     * Format config given in params
     *
     * @param string $host
     * @param string $port
     * @param string $chain
     * @param string $shop
     * @param string $user
     * @param string $password
     * @param bool   $ssl
     *
     * @return array
     */
    public function formatConfig($host, $port, $chain, $shop, $user, $password, $ssl)
    {
        return [
            'host'     => $host,
            'port'     => $port,
            'chain'    => $chain,
            'shop'     => $shop,
            'user'     => $user,
            'password' => $password,
            'ssl'      => $ssl
        ];
    }

    /**
     * Validate config
     *
     * @param $config
     *
     * @return bool
     */
    protected function validateConfig($config)
    {
        return $this->isConfigFieldSet('host', $config)
            && $this->isConfigFieldSet('port', $config)
            && $this->isConfigFieldSet('chain', $config)
            && $this->isConfigFieldSet('shop', $config)
            && $this->isConfigFieldSet('user', $config)
            && $this->isConfigFieldSet('password', $config)
            && $this->isConfigFieldSet('ssl', $config);
    }

    /**
     * Check if config is ready to generate URL
     * So if host, port and protocol are known
     *
     * @param array $config
     *
     * @return bool
     */
    public function isConfigReadyForUrl($config)
    {
        return $this->isConfigFieldSet('host', $config)
            && $this->isConfigFieldSet('port', $config)
            && $this->isConfigFieldSet('ssl', $config);
    }

    /**
     * Check if a specific config field is set and not null
     *
     * @param string $field
     * @param array  $config
     *
     * @return bool
     */
    protected function isConfigFieldSet($field, $config)
    {
        return array_key_exists($field, $config) && $config[$field] !== null;
    }

    /**
     * Returns data source (given by config field)
     *
     * @return string
     */
    public function getDataSource()
    {
        return 'fastmag_' . strtolower($this->getChain());
    }
}
