<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-22
 ******************************************************************************/

namespace Fastmag\Sync\Setup;

use Magento\Framework\DB\Adapter\AdapterInterface;
use Magento\Framework\DB\Ddl\Table;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\SchemaSetupInterface;
use Magento\Framework\Setup\UpgradeSchemaInterface;
use Zend_Db_Exception;

/**
 * Class UpgradeSchema
 *
 * Next schema installers
 */
class UpgradeSchema implements UpgradeSchemaInterface
{
    /**
     * Main method
     *
     * @param SchemaSetupInterface   $setup
     * @param ModuleContextInterface $context
     *
     * @throws Zend_Db_Exception
     */
    public function upgrade(SchemaSetupInterface $setup, ModuleContextInterface $context)
    {
        $setup->startSetup();

        if (version_compare($context->getVersion(), '0.1.1', '<')) {
            $this->addPaymentCodeRulesTable($setup);
        } elseif (version_compare($context->getVersion(), '0.1.2', '<')) {
            $this->addOrderTransactionRulesTable($setup);
        } elseif (version_compare($context->getVersion(), '0.2.0', '<')) {
            $this->addSynchronizedOrdersTable($setup);
            $this->addSynchronizedCustomersTable($setup);
        } elseif (version_compare($context->getVersion(), '0.3.0', '<')) {
            $this->addJobQueueToFastmagTable($setup);
        } elseif (version_compare($context->getVersion(), '0.3.1', '<')) {
            $this->addLogTable($setup);
        } elseif (version_compare($context->getVersion(), '0.3.2', '<')) {
            $this->addJobQueueToMagentoTable($setup);
            $this->addJobQueueToMagentoScheduledTable($setup);
        } elseif (version_compare($context->getVersion(), '0.5.0', '<')) {
            $this->addJobLastExecutionTable($setup);
        }
    }

    /**
     * Add paymentcode rules tables
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addPaymentCodeRulesTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_rule_paymentcode');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'rule_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Rule ID'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->addColumn(
                    'updated_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT_UPDATE],
                    'Updated at'
                )
                ->addColumn(
                    'payment_method',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Payment Method'
                )
                ->addColumn(
                    'fastmag_code',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'unsigned' => true,
                        'nullable' => false,
                        'default'  => 0
                    ],
                    'Fastmag Payment Code'
                )
                ->addColumn(
                    'card_type',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Card type'
                )
                ->setComment('Fastmag Sync Payment Code Binding Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add OrderTransaction rules table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addOrderTransactionRulesTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_rule_ordertransaction');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'rule_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Rule ID'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->addColumn(
                    'updated_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT_UPDATE],
                    'Updated at'
                )
                ->addColumn(
                    'payment_method',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Payment Method'
                )
                ->addColumn(
                    'order_status',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Order Status'
                )
                ->addColumn(
                    'fastmag_transaction_type',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Fastmag Transaction Type'
                )
                ->setComment('Fastmag Sync Order Transaction Binding Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add synchronized orders table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addSynchronizedOrdersTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_order');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'entity_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Line ID'
                )
                ->addColumn(
                    'order_id',
                    Table::TYPE_INTEGER,
                    null,
                    ['unsigned' => true, 'nullable' => false],
                    'Magento Order ID'
                )
                ->addColumn(
                    'transaction_id',
                    Table::TYPE_INTEGER,
                    null,
                    ['nullable' => false],
                    'Fastmag Transaction ID'
                )
                ->addColumn(
                    'transaction_id_2',
                    Table::TYPE_INTEGER,
                    null,
                    [],
                    'Fastmag Transaction ID 2'
                )
                ->addColumn(
                    'type',
                    Table::TYPE_TEXT,
                    30,
                    ['nullable' => false],
                    'Type'
                )
                ->addColumn(
                    'state',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'State'
                )
                ->addColumn(
                    'edi_csv',
                    Table::TYPE_BLOB,
                    null,
                    ['nullable' => false],
                    'EDI'
                )
                ->addColumn(
                    'edi_serialized',
                    Table::TYPE_BLOB,
                    null,
                    [],
                    'EDI serialized array'
                )
                ->addColumn(
                    'result_log',
                    Table::TYPE_BLOB,
                    null,
                    ['nullable' => false],
                    'Result'
                )
                ->addColumn(
                    'request_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Request Time'
                )
                ->addColumn(
                    'result_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Result Time'
                )
                ->addColumn(
                    'origin_shop',
                    Table::TYPE_TEXT,
                    255,
                    [],
                    'Origin shop'
                )
                ->addColumn(
                    'target_shop',
                    Table::TYPE_TEXT,
                    255,
                    [],
                    'Target shop'
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['state'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['state']
                )
                ->setComment('Magento Orders To Fastmag Transaction Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add synchronized customers table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addSynchronizedCustomersTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_customer');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'sync_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Synchronization ID'
                )
                ->addColumn(
                    'magento_customer_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'unsigned' => true,
                        'nullable' => false
                    ],
                    'Magento Customer ID'
                )
                ->addColumn(
                    'fastmag_customer_id',
                    Table::TYPE_INTEGER,
                    null,
                    ['nullable' => false],
                    'Fastmag Customer ID'
                )
                ->addColumn(
                    'is_deleted',
                    Table::TYPE_BOOLEAN,
                    null,
                    ['nullable' => false, 'default' => false],
                    'Is deleted in Magento and Fastmag'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->addColumn(
                    'processed_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Processed at'
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['magento_customer_id'],
                        AdapterInterface::INDEX_TYPE_UNIQUE
                    ),
                    ['magento_customer_id']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['fastmag_customer_id'],
                        AdapterInterface::INDEX_TYPE_UNIQUE
                    ),
                    ['fastmag_customer_id']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['created_at'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['created_at']
                )
                ->addForeignKey(
                    $setup->getFkName(
                        $tableName,
                        'magento_customer_id',
                        $setup->getTable('customer_entity'),
                        'entity_id'
                    ),
                    'magento_customer_id',
                    $setup->getTable('customer_entity'),
                    'entity_id',
                    Table::ACTION_CASCADE
                )
                ->setComment('Magento Customers To Fastmag Transaction Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add job queue to Fastmag table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addJobQueueToFastmagTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_jobqueue_to_fastmag');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'job_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Job ID'
                )
                ->addColumn(
                    'content_id',
                    Table::TYPE_TEXT,
                    32,
                    ['nullable' => false],
                    'Content ID'
                )
                ->addColumn(
                    'job_code',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Job Code'
                )
                ->addColumn(
                    'status',
                    Table::TYPE_BOOLEAN,
                    null,
                    ['nullable' => false, 'default' => 1],
                    'Content ID'
                )
                ->addColumn(
                    'retry_count',
                    Table::TYPE_INTEGER,
                    1,
                    ['default' => 0],
                    'Retry count'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->addColumn(
                    'processed_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    [],
                    'Processed at'
                )
                ->addColumn(
                    'hydrated_data',
                    Table::TYPE_BLOB,
                    '16M',
                    [],
                    'Hydrated data'
                )
                ->addColumn(
                    'message',
                    Table::TYPE_BLOB,
                    '16M',
                    [],
                    'Error message'
                )
                ->addColumn(
                    'trace',
                    Table::TYPE_BLOB,
                    '16M',
                    [],
                    'Error trace'
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['job_code'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['job_code']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['processed_at'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['processed_at']
                )
                ->setComment('Magento to Fastmag Job Queue Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add log table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addLogTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_log');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'log_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Log ID'
                )
                ->addColumn(
                    'message',
                    Table::TYPE_BLOB,
                    null,
                    [],
                    'Content ID'
                )
                ->addColumn(
                    'level',
                    Table::TYPE_INTEGER,
                    1,
                    ['nullable' => true, 'default' => 0],
                    'Log Level'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->setComment('Fastmag Sync Log Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add jobs queue to Magento table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addJobQueueToMagentoTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_jobqueue_to_magento');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'job_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Job ID'
                )
                ->addColumn(
                    'content_id',
                    Table::TYPE_TEXT,
                    60,
                    [],
                    'Content ID'
                )
                ->addColumn(
                    'job_code',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Job Code'
                )
                ->addColumn(
                    'content',
                    Table::TYPE_BLOB,
                    null,
                    [],
                    'Object Content'
                )
                ->addColumn(
                    'status',
                    Table::TYPE_BOOLEAN,
                    null,
                    ['nullable' => false, 'default' => 1],
                    'Content ID'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->addColumn(
                    'scheduled_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    [],
                    'Scheduled at'
                )
                ->addColumn(
                    'processed_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    [],
                    'Processed at'
                )
                ->addColumn(
                    'comment',
                    Table::TYPE_BLOB,
                    null,
                    [],
                    'Comment'
                )
                ->addColumn(
                    'priority',
                    Table::TYPE_INTEGER,
                    8,
                    ['nullable' => false, 'unsigned' => true, 'default' => 0],
                    'Priority'
                )
                ->addColumn(
                    'hydrated_data',
                    Table::TYPE_BLOB,
                    '16M',
                    [],
                    'Hydrated data'
                )
                ->addColumn(
                    'message',
                    Table::TYPE_BLOB,
                    '16M',
                    [],
                    'Error message'
                )
                ->addColumn(
                    'trace',
                    Table::TYPE_BLOB,
                    '16M',
                    [],
                    'Error trace'
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['content_id'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['content_id']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['job_code'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['job_code']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['scheduled_at'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['scheduled_at']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['processed_at'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['processed_at']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['priority'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['priority']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['content_id', 'job_code', 'priority'],
                        AdapterInterface::INDEX_TYPE_UNIQUE
                    ),
                    ['content_id', 'job_code', 'priority'],
                    ['type' => AdapterInterface::INDEX_TYPE_UNIQUE]
                )
                ->setComment('Fastmag to Magento Job Queue Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add scheduled jobs queue to Magento table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addJobQueueToMagentoScheduledTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_jobqueue_to_magento_scheduled');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'job_id',
                    Table::TYPE_INTEGER,
                    null,
                    [
                        'identity' => true,
                        'unsigned' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Job ID'
                )
                ->addColumn(
                    'content_id',
                    Table::TYPE_TEXT,
                    60,
                    [],
                    'Content ID'
                )
                ->addColumn(
                    'job_code',
                    Table::TYPE_TEXT,
                    255,
                    ['nullable' => false],
                    'Job Code'
                )
                ->addColumn(
                    'fastmag_shop',
                    Table::TYPE_TEXT,
                    30,
                    [],
                    'Fastmag Shop'
                )
                ->addColumn(
                    'content',
                    Table::TYPE_BLOB,
                    null,
                    [],
                    'Content'
                )
                ->addColumn(
                    'created_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT],
                    'Created at'
                )
                ->addColumn(
                    'scheduled_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    [],
                    'Scheduled at'
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['content_id'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['content_id']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['job_code'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['job_code']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['fastmag_shop'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['fastmag_shop']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['scheduled_at'],
                        AdapterInterface::INDEX_TYPE_INDEX
                    ),
                    ['scheduled_at']
                )
                ->addIndex(
                    $setup->getIdxName(
                        $tableName,
                        ['content_id', 'job_code', 'fastmag_shop', 'scheduled_at'],
                        AdapterInterface::INDEX_TYPE_UNIQUE
                    ),
                    ['content_id', 'job_code', 'fastmag_shop', 'scheduled_at'],
                    ['type' => AdapterInterface::INDEX_TYPE_UNIQUE]
                )
                ->setComment('Scheduled Fastmag to Magento Job Queue Table');

            $setup->getConnection()->createTable($table);
        }
    }

    /**
     * Add jobs' last executions table
     *
     * @param SchemaSetupInterface $setup
     *
     * @throws Zend_Db_Exception
     */
    public function addJobLastExecutionTable(SchemaSetupInterface $setup)
    {
        $tableName = $setup->getTable('fastmag_sync_job_last_execution');

        if ($setup->getConnection()->isTableExists($tableName) !== true) {
            $table = $setup->getConnection()
                ->newTable($tableName)
                ->addColumn(
                    'job_code',
                    Table::TYPE_TEXT,
                    255,
                    [
                        'identity' => true,
                        'nullable' => false,
                        'primary'  => true
                    ],
                    'Job code'
                )
                ->addColumn(
                    'last_executed_at',
                    Table::TYPE_TIMESTAMP,
                    null,
                    ['nullable' => false, 'default' => Table::TIMESTAMP_INIT_UPDATE],
                    'Last executed at'
                )
                ->setComment('Job Last Execution Table');

            $setup->getConnection()->createTable($table);
        }
    }
}
