<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-09-25
 ******************************************************************************/

namespace Fastmag\Sync\Controller\Adminhtml\Preconfig;

use Exception;
use Fastmag\Sync\Controller\Adminhtml\Preconfig;
use Fastmag\Sync\FastmagSetup\Uninstaller;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\System\Connection\Edi\Config;
use Fastmag\Sync\Model\System\Preconfig\File;
use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Request\Http;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Math\Random;
use Magento\Framework\Module\ModuleResource;
use Zend_Mail;

/**
 * Class Generate
 *
 * Preconfig generate SQL action
 */
class Generate extends Preconfig
{
    /** @var ModuleResource $moduleResource */
    protected $moduleResource;

    /** @var Random $randomGenerator */
    protected $randomGenerator;

    /** @var Uninstaller $uninstaller */
    protected $uninstaller;

    /** @var ResourceConnection $resourceConnection */
    protected $resourceConnection;

    /** @var Config $ediConfig */
    protected $ediConfig;

    /** @var JsonFactory $resultJsonFactory */
    protected $resultJsonFactory;

    /** @var File $file */
    protected $file;

    /** @var string $user */
    protected $user;

    /** @var string $pass */
    protected $pass;

    /** @var string $toFile */
    protected $toFile;

    /**
     * Generate constructor
     *
     * @param Context              $context
     * @param ScopeConfigInterface $scopeConfig
     * @param Logger               $logger
     * @param Zend_Mail            $zendMail
     * @param ModuleResource       $moduleResource
     * @param Random               $randomGenerator
     * @param Uninstaller          $uninstaller
     * @param ResourceConnection   $resourceConnection
     * @param Config               $ediConfig
     * @param File                 $file
     * @param JsonFactory          $resultJsonFactory
     */
    public function __construct(
        Context $context,
        ScopeConfigInterface $scopeConfig,
        Logger $logger,
        Zend_Mail $zendMail,
        ModuleResource $moduleResource,
        Random $randomGenerator,
        Uninstaller $uninstaller,
        ResourceConnection $resourceConnection,
        Config $ediConfig,
        File $file,
        JsonFactory $resultJsonFactory
    ) {
        parent::__construct($context, $scopeConfig, $logger, $zendMail);

        $this->moduleResource = $moduleResource;
        $this->randomGenerator = $randomGenerator;
        $this->uninstaller = $uninstaller;
        $this->resourceConnection = $resourceConnection;
        $this->ediConfig = $ediConfig;
        $this->file = $file;
        $this->resultJsonFactory = $resultJsonFactory;
    }

    /**
     * @inheritDoc
     *
     * @throws LocalizedException
     */
    public function execute()
    {
        $this->user = $this->getRequest()->getParam('user');
        $this->pass = str_shuffle($this->randomGenerator->getRandomString(12) . '@#/~');

        $version = $this->moduleResource->getDbVersion('Fastmag_Sync');

        $proxyIp = $this->scopeConfig->getValue('fastmag_sync_connect/proxy/host');
        $proxyUser = $this->scopeConfig->getValue('fastmag_sync_connect/proxy/user');
        $proxyPass = $this->scopeConfig->getValue('fastmag_sync_connect/proxy/password');

        $dataSource = $this->ediConfig->getDataSource();
        $sql = $this->uninstaller->getUninstallQueries();
        $entityTypes = array_keys($sql);

        $result = '/* Version module HomeMade : ' . $version . ' */<br />'
            . str_replace(PHP_EOL, '<br />', parent::getEmailContent()) . '<br /><br />'
            . '<hr />'
            . '<h3>/****** <strong>[Si ancien module]</strong> Nettoyage sur Fastmag ******/' . '</h3><br />';

        $grantList = [];
        foreach ($entityTypes as $type) {
            foreach ($sql[$type] as $name => $query) {
                $result .= $query . '<br />';

                if ($type === 'tables') {
                    $grantList[] = 'GRANT ALL PRIVILEGES ON ' . $dataSource . '.'
                        . str_replace('DROP TABLE IF EXISTS ', '', $query)
                        . ' TO \'' . $this->user . '\'@\'' . $proxyIp . '\';';
                }
            }
        }

        $grantList[] = 'GRANT TRIGGER ON ' . $dataSource . '.* TO \'' . $this->user . '\'@\'' . $proxyIp . '\';';

        $result .= '<br />';
        $result .= '<h3>/****** <strong>[Si ancien compte]</strong> Suppression anciens droits ******/</h3>' . '<br />';
        $result .= 'REVOKE ALL PRIVILEGES, GRANT OPTION FROM \'' . $this->user . '\'@\'%\';' . '<br />';
        $result .= 'REVOKE ALL PRIVILEGES, GRANT OPTION FROM \'' . $this->user . '\'@\'' . $proxyIp . '\';' . '<br />';

        $result .= '<h3>/****** <strong>[Si anciennes tables]</strong> Drop ******/</h3>' . '<br />';
        $result .= 'DROP TABLE IF EXISTS presta_job_queue;' . '<br />';
        $result .= 'DROP TABLE IF EXISTS presta_job_queue_schedule;' . '<br />';
        $result .= 'DROP TABLE IF EXISTS presta_job_queue_website;' . '<br />';

        $result .= '<br />';
        $result .= '<h3>/********************* <strong>Nouveaux Droits</strong> *********************/</h3>' . '<br />';
        $result .= '/****** Creation user ******/' . '<br />';
        $result .= 'CREATE USER ' . $this->user . '@' . $proxyIp . ' IDENTIFIED BY \'' . $this->pass . '\';' . '<br />';

        $result .= '<br />';
        $result .= '<h3>/****** Privileges Proxy tables module / trigger ******/</h3>' . '<br />';
        $result .= implode('<br />', $grantList) . '<br />';

        $result .= '<br />';
        $result .= '<h3>/****** Privileges Proxy SELECT ******/</h3>' . '<br />';
        $result .= 'GRANT SELECT ON ' . $dataSource . '.* TO \'' . $this->user . '\'@\'' . $proxyIp . '\';' . '<br />';
        $result .= '<br />';

        // Home-Made Proxy
        $result .= '<h3>/****** Privileges Proxy Monitoring ******/</h3>';
        $result .= '<br />/* Proxy Monitor */<br />';
        $result .= 'GRANT USAGE ON ' . $dataSource . '.* TO \'' . $proxyUser . '\'@\'' . $proxyIp
            . '\' IDENTIFIED BY \'' . $proxyPass . '\';' . '<br />';
        $result .= '<br / >';
        $result .= 'FLUSH PRIVILEGES;' . '<br />';

        $this->sendQueriesEmail();

        $this->file->write($this->toFile);

        if (!$this->file->exists()) {
            $result .= '<br /><br />############################################################################<br />';
            $result .= nl2br($this->toFile);
            $result .= '<br />############################################################################<br /><br />';
        }

        $resultJson = $this->resultJsonFactory->create();

        return $resultJson->setData([$result]);
    }

    /**
     * @inheritDoc
     */
    protected function getEmailContent()
    {
        $content = parent::getEmailContent();

        $connection = $this->resourceConnection->getConnection();
        $hashPass = $connection->fetchOne('SELECT PASSWORD(\'' . $this->pass . '\') AS Pwd');

        $dataSource = $this->ediConfig->getDataSource();

        /** @var Http $request */
        $request = $this->getRequest();
        $externalIp = gethostbyname($request->getServer('HTTP_HOST'));

        $toProxy[] = '/* Proxy Config */';
        $toProxy[] = 'INSERT INTO mysql_servers (hostgroup_id, hostname, port)
            VALUES ({hostgroup}, \'{server_fastmag}\', 3306);';
        $toProxy[] = 'INSERT INTO mysql_users
                (username, password, default_hostgroup, default_schema, transaction_persistent)
            VALUES (\'' . $this->user . '\', \'' . $hashPass . '\', {hostgroup}, \'' . $dataSource . '\', 0);';
        $toProxy[] = 'INSERT INTO mysql_query_rules
                (rule_id, active, username, schemaname, match_digest, destination_hostgroup, apply)
            VALUES (NULL, 1, \'' . $this->user . '\', \'' . $dataSource . '\', \'.*\', {hostgroup}, 1);';

        $toProxy[] = 'LOAD MYSQL SERVERS TO RUNTIME;';
        $toProxy[] = 'SAVE MYSQL SERVERS TO DISK;';
        $toProxy[] = 'LOAD MYSQL USERS TO RUNTIME;';
        $toProxy[] = 'SAVE MYSQL USERS TO DISK;';
        $toProxy[] = 'LOAD MYSQL QUERY RULES TO RUNTIME;';
        $toProxy[] = 'SAVE MYSQL QUERY RULES TO DISK;';
        $toProxy[] = '';
        $toProxy[] = '/* Iptables */';
        $toProxy[] = '## ' . ucfirst(strtolower($this->ediConfig->getChain()));
        $toProxy[] = 'iptables -t filter -A INPUT -p tcp --dport 3342 -s ' . $request->getServer('SERVER_ADDR')
            . ' -j ACCEPT';
        $toProxy[] = '### OU SI DIFFERENTE ';
        $toProxy[] = 'iptables -t filter -A INPUT -p tcp --dport 3342 -s ' . $externalIp . ' -j ACCEPT';

        $this->toFile = 'User password "' . $this->pass . '" - ' . date('Y-m-d H:i:s')
            . PHP_EOL . PHP_EOL . implode(PHP_EOL, $toProxy);

        $content .= PHP_EOL . '----------------------------------------' . PHP_EOL . $this->toFile . PHP_EOL . PHP_EOL;

        return $content;
    }

    /**
     * Send queries email for monitoring
     *
     * @return void
     */
    protected function sendQueriesEmail()
    {
        /** @var Http $request */
        $request = $this->getRequest();

        $httpHost = $request->getServer('HTTP_HOST');
        $date = date('Y-m-d H:i:s');
        $subject = 'Accès ace_preconfig / ' . $httpHost . ' / ' . $date . ' / TMP Proxy';

        $emailFrom = $this->scopeConfig->getValue('trans_email/ident_general/email');
        $emailTo = 'monitoring@home-made.io';

        try {
            $this->sendEmail($emailFrom, $emailTo, $subject, $this->getEmailContent());
        } catch (Exception $e) {
            $this->logger->warning(
                'Error while sending preconfig access email : ' . $e->getMessage()
                . ' - Trace : ' . $e->getTraceAsString()
            );
        }
    }
}
