<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-08-26
 ******************************************************************************/

namespace Fastmag\Sync\Model;

use Exception;
use Fastmag\Sync\Api\Data\LogInterface;
use Fastmag\Sync\Api\Data\LogInterfaceFactory;
use Fastmag\Sync\Api\Data\LogSearchResultsInterfaceFactory;
use Fastmag\Sync\Api\LogRepositoryInterface;
use Fastmag\Sync\Model\ResourceModel\Log as ResourceModel;
use Fastmag\Sync\Model\ResourceModel\Log\CollectionFactory;
use Magento\Framework\Api\DataObjectHelper;
use Magento\Framework\Api\SearchCriteria\CollectionProcessor\FilterProcessor;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Reflection\DataObjectProcessor;

/**
 * Class LogRepository
 *
 * Repository class for log model
 */
class LogRepository implements LogRepositoryInterface
{
    /** @var ResourceModel $resource */
    protected $resource;

    /** @var LogFactory $logFactory */
    protected $logFactory;

    /** @var CollectionFactory $collectionFactory */
    protected $collectionFactory;

    /** @var LogSearchResultsInterfaceFactory $searchResultsFactory */
    protected $searchResultsFactory;

    /** @var DataObjectHelper $dataObjectHelper */
    protected $dataObjectHelper;

    /** @var DataObjectProcessor $dataObjectProcessor */
    protected $dataObjectProcessor;

    /** @var LogInterfaceFactory $logInterfaceFactory */
    protected $logInterfaceFactory;

    /** @var CollectionProcessorInterface $collectionProcessor */
    private $collectionProcessor;

    /**
     * LogRepository constructor
     *
     * @param ResourceModel                     $resource
     * @param LogFactory                        $logFactory
     * @param LogInterfaceFactory               $logInterfaceFactory
     * @param CollectionFactory                 $collectionFactory
     * @param LogSearchResultsInterfaceFactory  $searchResultsFactory
     * @param DataObjectHelper                  $dataObjectHelper
     * @param DataObjectProcessor               $dataObjectProcessor
     * @param CollectionProcessorInterface|null $collectionProcessor
     */
    public function __construct(
        ResourceModel $resource,
        LogFactory $logFactory,
        LogInterfaceFactory $logInterfaceFactory,
        CollectionFactory $collectionFactory,
        LogSearchResultsInterfaceFactory $searchResultsFactory,
        DataObjectHelper $dataObjectHelper,
        DataObjectProcessor $dataObjectProcessor,
        CollectionProcessorInterface $collectionProcessor = null
    ) {
        $this->resource = $resource;
        $this->logFactory = $logFactory;
        $this->collectionFactory = $collectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->dataObjectHelper = $dataObjectHelper;
        $this->logInterfaceFactory = $logInterfaceFactory;
        $this->dataObjectProcessor = $dataObjectProcessor;
        $this->collectionProcessor = $collectionProcessor ?: $this->getCollectionProcessor();
    }

    /**
     * @inheritDoc
     */
    public function save(LogInterface $log)
    {
        try {
            $this->resource->save($log);
        } catch (Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $log;
    }

    /**
     * @inheritDoc
     */
    public function getById($logId)
    {
        $log = $this->logFactory->create();
        $this->resource->load($log, $logId);
        if (!$log->getId()) {
            throw new NoSuchEntityException(__('The log with the "%1" ID doesn\'t exist.', $logId));
        }

        return $log;
    }

    /**
     * @inheritDoc
     */
    public function getList(SearchCriteriaInterface $criteria)
    {
        $collection = $this->collectionFactory->create();

        $this->collectionProcessor->process($criteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($criteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * @inheritDoc
     */
    public function delete(LogInterface $log)
    {
        try {
            $this->resource->delete($log);
        } catch (Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById($entityId)
    {
        return $this->delete($this->getById($entityId));
    }

    /**
     * Retrieve collection processor
     *
     * @deprecated
     *
     * @return CollectionProcessorInterface
     */
    private function getCollectionProcessor()
    {
        if (!$this->collectionProcessor) {
            $this->collectionProcessor = ObjectManager::getInstance()->get(FilterProcessor::class);
        }

        return $this->collectionProcessor;
    }
}
