<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-10-12
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Entity\ToFastmag;

use Fastmag\Sync\Api\Data\OrderInterface as SyncOrder;
use Fastmag\Sync\Api\Data\Rule\StoresellerInterface as StoresellerRule;
use Fastmag\Sync\Model\Process\EntityInterface;
use Fastmag\Sync\Model\Process\Entity\ToFastmag\Order\Item as OrderItemEntity;
use Fastmag\Sync\Model\Process\Entity\ToFastmag\Order\ItemFactory as OrderItemEntityFactory;
use Magento\Sales\Model\Order\Item;

/**
 * Class Order
 *
 * Entity class for order.
 * Save and transform data from Magento for Fastmag easy integration,
 * and transfer it from hydration worker to integration worker
 */
class Order implements EntityInterface
{
    /** @var int $magentoId */
    protected $magentoId;

    /** @var int $incrementId */
    protected $incrementId;

    /** @var string $status */
    protected $status;

    /** @var int $totalQty */
    protected $totalQty;

    /** @var bool $isExcludedTax */
    protected $isExcludedTax;

    /** @var float $grandTotal */
    protected $grandTotal;

    /** @var OrderItemEntity[] $items */
    protected $items = [];

    /** @var float $shippingRate */
    protected $shippingRate;

    /** @var float $shippingDiscount */
    protected $shippingDiscount;

    /** @var string $paymentId */
    protected $paymentId;

    /** @var string $paymentCode */
    protected $paymentCode;

    /** @var StoresellerRule $storesellerRule */
    protected $storesellerRule;

    /** @var string $transactionType */
    protected $transactionType;

    /** @var SyncOrder $transactionType */
    protected $lastTransaction;

    /** @var bool $ediFailed */
    protected $ediFailed;

    /** @var int $customerId */
    protected $customerId;

    /** @var Customer $customer */
    protected $customer;

    /** @var Address $billingAddress */
    protected $billingAddress;

    /** @var Address $shippingAddress */
    protected $shippingAddress;

    /** @var OrderItemEntityFactory $itemEntityFactory */
    protected $itemEntityFactory;

    /**
     * Order constructor.
     *
     * @param OrderItemEntityFactory $itemEntityFactory
     */
    public function __construct(OrderItemEntityFactory $itemEntityFactory)
    {
        $this->itemEntityFactory = $itemEntityFactory;
    }

    /**
     * Get entity ID
     *
     * @return int
     */
    public function getMagentoId()
    {
        return $this->magentoId;
    }

    /**
     * Get increment ID
     *
     * @return int
     */
    public function getIncrementId()
    {
        return $this->incrementId;
    }

    /**
     * Get status
     *
     * @return string
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Get total qty
     *
     * @return int
     */
    public function getTotalQty()
    {
        return $this->totalQty;
    }

    /**
     * Is order excluded tax
     *
     * @return bool
     */
    public function isExcludedTax()
    {
        return $this->isExcludedTax;
    }

    /**
     * Get grand total
     *
     * @return float
     */
    public function getGrandTotal()
    {
        return $this->grandTotal;
    }

    /**
     * Get order items
     *
     * @return OrderItemEntity[]
     */
    public function getItems()
    {
        return $this->items;
    }

    /**
     * Get shipping rate
     *
     * @return float
     */
    public function getShippingRate()
    {
        return $this->shippingRate;
    }

    /**
     * Get shipping discount
     *
     * @return float
     */
    public function getShippingDiscount()
    {
        return $this->shippingDiscount;
    }

    /**
     * Get payment ID
     *
     * @return string
     */
    public function getPaymentId()
    {
        return $this->paymentId;
    }

    /**
     * Get payment code
     *
     * @return string
     */
    public function getPaymentCode()
    {
        return $this->paymentCode;
    }

    /**
     * Get store/seller rule
     *
     * @return StoresellerRule|null
     */
    public function getStoresellerRule()
    {
        return $this->storesellerRule;
    }

    /**
     * Get transaction type
     *
     * @return string
     */
    public function getTransactionType()
    {
        return $this->transactionType;
    }

    /**
     * Get last transaction
     *
     * @return SyncOrder|null
     */
    public function getLastTransaction()
    {
        return $this->lastTransaction;
    }

    /**
     * Is EDI failed
     *
     * @return bool
     */
    public function isEdiFailed()
    {
        return $this->ediFailed;
    }

    /**
     * Get customer ID
     *
     * @return int
     */
    public function getCustomerId()
    {
        return $this->customerId;
    }

    /**
     * Get customer
     *
     * @return Customer
     */
    public function getCustomer()
    {
        return $this->customer;
    }

    /**
     * Get billing address
     *
     * @return Address|null
     */
    public function getBillingAddress()
    {
        return $this->billingAddress;
    }

    /**
     * Get shipping address
     *
     * @return Address|null
     */
    public function getShippingAddress()
    {
        return $this->shippingAddress;
    }

    /**
     * Set Magento ID
     *
     * @param int $magentoId
     *
     * @return Order
     */
    public function setMagentoId($magentoId)
    {
        $this->magentoId = $magentoId;

        return $this;
    }

    /**
     * Set increment ID
     *
     * @param int $incrementId
     *
     * @return Order
     */
    public function setIncrementId($incrementId)
    {
        $this->incrementId = $incrementId;

        return $this;
    }

    /**
     * Set status
     *
     * @param string $status
     *
     * @return Order
     */
    public function setStatus($status)
    {
        $this->status = $status;

        return $this;
    }

    /**
     * Set total quantity
     *
     * @param int $totalQty
     *
     * @return Order
     */
    public function setTotalQty($totalQty)
    {
        $this->totalQty = $totalQty;

        return $this;
    }

    /**
     * Set is excluded tax boolean
     *
     * @param bool $isExcludedTax
     *
     * @return Order
     */
    public function setIsExcludedTax($isExcludedTax)
    {
        $this->isExcludedTax = $isExcludedTax;

        return $this;
    }

    /**
     * Set grand total
     *
     * @param float $grandTotal
     *
     * @return Order
     */
    public function setGrandTotal($grandTotal)
    {
        $this->grandTotal = $grandTotal;

        return $this;
    }

    /**
     * Set items
     *
     * @param OrderItemEntity[] $items
     *
     * @return Order
     */
    public function setItems($items)
    {
        $this->items = $items;

        return $this;
    }

    /**
     * Set shipping rate
     *
     * @param float $shippingRate
     *
     * @return Order
     */
    public function setShippingRate($shippingRate)
    {
        $this->shippingRate = $shippingRate;

        return $this;
    }

    /**
     * Set shipping discount
     *
     * @param float $shippingDiscount
     *
     * @return Order
     */
    public function setShippingDiscount($shippingDiscount)
    {
        $this->shippingDiscount = $shippingDiscount;

        return $this;
    }

    /**
     * Set payment ID
     *
     * @param string $paymentId
     *
     * @return Order
     */
    public function setPaymentId($paymentId)
    {
        $this->paymentId = $paymentId;

        return $this;
    }

    /**
     * Set payment code
     *
     * @param string $paymentCode
     *
     * @return Order
     */
    public function setPaymentCode($paymentCode)
    {
        $this->paymentCode = $paymentCode;

        return $this;
    }

    /**
     * Set store/seller rule
     *
     * @param StoresellerRule $storesellerRule
     *
     * @return Order
     */
    public function setStoresellerRule($storesellerRule)
    {
        $this->storesellerRule = $storesellerRule;

        return $this;
    }

    /**
     * Set transaction type
     *
     * @param string $transactionType
     *
     * @return Order
     */
    public function setTransactionType($transactionType)
    {
        $this->transactionType = $transactionType;

        return $this;
    }

    /**
     * Set last transaction
     *
     * @param SyncOrder $lastTransaction
     *
     * @return Order
     */
    public function setLastTransaction($lastTransaction)
    {
        $this->lastTransaction = $lastTransaction;

        return $this;
    }

    /**
     * Set EDI failed
     *
     * @param bool $ediFailed
     *
     * @return Order
     */
    public function setEdiFailed($ediFailed)
    {
        $this->ediFailed = $ediFailed;

        return $this;
    }

    /**
     * Set customer ID
     *
     * @param int $customerId
     *
     * @return Order
     */
    public function setCustomerId($customerId)
    {
        $this->customerId = $customerId;

        return $this;
    }

    /**
     * Set customer
     *
     * @param Customer $customer
     *
     * @return Order
     */
    public function setCustomer($customer)
    {
        $this->customer = $customer;

        return $this;
    }

    /**
     * Set billing address
     *
     * @param Address $address
     *
     * @return Order
     */
    public function setBillingAddress($address)
    {
        $this->billingAddress = $address;

        return $this;
    }

    /**
     * Set shipping address
     *
     * @param Address $address
     *
     * @return Order
     */
    public function setShippingAddress($address)
    {
        $this->shippingAddress = $address;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        $itemsData = '';
        if (count($this->getItems()) > 0) {
            foreach ($this->getItems() as $item) {
                $itemsData[$item->getMagentoId()] = $item->export();
            }
        }

        return [
            'order' => [
                'order_id'          => $this->getMagentoId(),
                'increment_id'      => $this->getIncrementId(),
                'status'            => $this->getStatus(),
                'total_qty  '       => $this->getTotalQty(),
                'is_excluding_tax'  => $this->isExcludedTax(),
                'items'             => $itemsData,
                'shipping_rate'     => $this->getShippingRate(),
                'shipping_discount' => $this->getShippingDiscount(),
                'payment_id'        => $this->getPaymentId(),
                'payment_code'      => $this->getPaymentCode(),
                'customer'          => ($this->getCustomer() ? $this->getCustomer()->export() : null),
                'addresses'         => [
                    'billing'  => ($this->getBillingAddress() ? $this->getBillingAddress()->export() : null),
                    'shipping' => ($this->getShippingAddress() ? $this->getShippingAddress()->export() : null)
                ]
            ],
            'edi'   => [
                'chain'                 => ($this->getStoresellerRule() ?
                    $this->getStoresellerRule()->getChain() : null),
                'fastmag_shop'          => ($this->getStoresellerRule() ?
                    $this->getStoresellerRule()->getFastmagShop() : null),
                'seller'                => ($this->getStoresellerRule() ?
                    $this->getStoresellerRule()->getSeller() : null),
                'stocks'                => ($this->getStoresellerRule() ?
                    $this->getStoresellerRule()->getReferenceStock() : null),
                'transaction_type'      => $this->getTransactionType(),
                'last_transaction_id'   => ($this->getLastTransaction() ? $this->getLastTransaction()->getId() : null),
                'last_transaction_type' => ($this->getLastTransaction() ?
                    $this->getLastTransaction()->getTransactionType() : null),
                'edi_failure'           => $this->isEdiFailed()
            ]
        ];
    }

    /**
     * Add item to the items list
     *
     * @param Item $item
     *
     * @return void
     */
    public function addItem($item)
    {
        $itemEntity = $this->itemEntityFactory->create();

        $itemEntity->setMagentoId($item->getItemId())
            ->setParentItemId($item->getParentItemId())
            ->setFastmagId($item->getFastmagProductId())
            ->setName($item->getName())
            ->setPriceInclTax($item->getBasePriceInclTax())
            ->setQtyOrdered($item->getQtyOrdered())
            ->setQtyCanceled($item->getQtyCanceled())
            ->setDiscountAmount($item->getBaseDiscountAmount())
            ->setRowTotalInclTax($item->getBaseRowTotalInclTax());

        $this->items[$itemEntity->getMagentoId()] = $itemEntity;
    }

    /**
     * Get item by Fastmag ID
     *
     * @param int $fastmagId
     *
     * @return OrderItemEntity|null
     */
    public function getItemByFastmagId($fastmagId)
    {
        $result = null;

        foreach ($this->getItems() as $itemId => $item) {
            if ($item->getFastmagId() === $fastmagId) {
                $result = $item;

                break;
            }
        }

        return $result;
    }

    /**
     * Get item by Magento ID
     *
     * @param int $magentoId
     *
     * @return OrderItemEntity|null
     */
    public function getItemByMagentoId($magentoId)
    {
        $result = null;

        if (array_key_exists($magentoId, $this->getItems())) {
            $result = $this->getItems()[$magentoId];
        }

        return $result;
    }
}
