<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-10-21
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Manager;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToFastmag as Job;
use Fastmag\Sync\Model\Jobqueue\ToFastmagRepository;
use Fastmag\Sync\Model\Process\Manager;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Clean;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration;
use Fastmag\Sync\Model\Process\WorkerFactory;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToFastmag\Collection;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class ToFastmag
 *
 * Workers manager for synchronization from Magento to Fastmag
 */
class ToFastmag extends Manager
{
    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var ToFastmagRepository $jobRepository */
    protected $jobRepository;

    /**
     * ToFastmag constructor
     *
     * @param Logger                $logger
     * @param Config                $config
     * @param WorkerFactory         $workerFactory
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SortOrderBuilder      $sortOrderBuilder
     * @param ToFastmagRepository   $jobRepository
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        ToFastmagRepository $jobRepository
    ) {
        parent::__construct($logger, $config, $workerFactory);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * Main method
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return void
     *
     * @throws Exception
     */
    public function run($jobCode, $jobId)
    {
        try {
            $checkOk = $this->checkPrerequisites($jobCode, $jobId);

            if ($checkOk) {
                $this->logger->debug('To Fastmag Sync - Begin process');

                $jobsCollectionByCode = $this->getJobsCollections($jobCode, $jobId);

                if (count($jobsCollectionByCode) > 0) {
                    foreach ($jobsCollectionByCode as $collectionJobCode => $jobsCollection) {
                        $this->processJobs($collectionJobCode, $jobsCollection);
                    }
                }

                /** @var Clean cleanWorker */
                $cleanWorker = $this->workerFactory->create('tofastmag_clean');
                $cleanWorker->run();
            }
        } catch (ProcessException $e) {
            $this->logger->critical($e->getMessage());

            throw $e;
        }
    }

    /**
     * Get array of jobs collection to process
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return array
     */
    protected function getJobsCollections($jobCode, $jobId)
    {
        $sortOrderJobId = $this->sortOrderBuilder->setField(ToFastmagInterface::JOB_ID)
            ->setAscendingDirection()
            ->create();

        $this->searchCriteriaBuilder->addFilter(ToFastmagInterface::STATUS, ToFastmagInterface::STATUS_OK)
            ->addFilter(ToFastmagInterface::PROCESSED_AT, null, 'null')
            ->addSortOrder($sortOrderJobId)
            ->setPageSize($this->getSyncLimit());

        if ($jobId) {
            $this->searchCriteriaBuilder->addFilter(ToFastmagInterface::JOB_ID, $jobId);
        } elseif ($jobCode) {
            $this->searchCriteriaBuilder->addFilter(ToFastmagInterface::JOB_CODE, $jobCode);
        }

        $searchCriteria = $this->searchCriteriaBuilder->create();

        return $this->jobRepository->getListByCode($searchCriteria);
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites($jobCode = null, $jobId = null)
    {
        $result = parent::checkPrerequisites();

        if ($result) {
            if ($jobId !== null) {
                try {
                    $job = $this->jobRepository->getById($jobId);
                    $jobCode = $job->getJobCode();
                } catch (Exception $e) {
                    throw new ProcessException('Job ID #' . $jobId . ' does not exist in queue.');
                }
            }

            if ($jobCode !== null) {
                /** @var Integration $worker */
                $worker = $this->workerFactory->create($jobCode);

                if (!$worker->isEnabled()) {
                    throw new ProcessException('Jobs "' . $jobCode . '" are not enabled.');
                }
            }
        }

        return $result;
    }

    /**
     * Process jobs collection
     *
     * @param string     $jobCode
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function processJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $this->resetJobsError($jobsCollection);

            /** @var Integration $worker */
            $worker = $this->workerFactory->create($jobCode);

            if ($worker->isEnabled()) {
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Beginning hydration');

                $this->hydrateJobs($jobCode, $jobsCollection);
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Hydration done, beginning integration.');

                $this->runWorker($worker, $jobsCollection);
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Integration done, beginning results handling');

                $this->handleResults($worker);
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Results handling done');
            } else {
                $this->logger->debug('Jobs "' . $worker->getCode() . '" are not enabled.');
            }
        }
    }

    /**
     * Reset jobs collection message and trace data
     *
     * @param Collection $jobsCollection
     *
     * @return void
     */
    protected function resetJobsError($jobsCollection)
    {
        foreach ($jobsCollection->getItems() as $key => $job) {
            $job->setMessage(null)
                ->setTrace(null);

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $e) {
                $jobsCollection->removeItemByKey($key);
            }
        }
    }

    /**
     * Run hydration workers to hydrate current jobs and add entities created to the jobs collection
     *
     * @param string     $jobCode
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function hydrateJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $hydrationJobCode = $this->getHydrationJobCode($jobCode);

            if ($hydrationJobCode) {
                /** @var Hydration $worker */
                $worker = $this->workerFactory->create($hydrationJobCode);

                /** @throws ProcessException */
                $this->runWorker($worker, $jobsCollection);
            }
        }
    }

    /**
     * Get hydration worker code for sync job code
     *
     * @todo make this method more generic
     *
     * @param string $jobCode
     *
     * @return string
     */
    protected function getHydrationJobCode($jobCode)
    {
        $result = '';

        if (stripos($jobCode, 'customer') !== false) {
            if (stripos($jobCode, 'delete') !== false) {
                $result = 'tofastmag_hydration_customer_delete';
            } else {
                $result = 'tofastmag_hydration_customer_save';
            }
        } elseif (stripos($jobCode, 'order') !== false) {
            $result = 'tofastmag_hydration_order';
        }

        return $result;
    }

    /**
     * Handle jobs results
     *
     * @param Integration $worker
     */
    protected function handleResults($worker)
    {
        $errors = $worker->getErrors();

        /** @var Job $job */
        foreach ($worker->getJobs() as $job) {
            if ($worker->isJobInError($job)) {
                $e = new Exception();
                $job->setStatus(Job::STATUS_ERROR)
                    ->setMessage($errors[$job->getId()])
                    ->setTrace($e->getTraceAsString());

                $this->logger->error(
                    'Error on job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId(). ': '
                    . $job->getMessage()
                );
            } else {
                $job->setStatus(Job::STATUS_OK)
                    ->setProcessedAt(date('Y-m-d H:i:s'));
            }

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $e) {
                $this->logger->critical(
                    'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                    . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                );
            }
        }
    }
}
