<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-10-21
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Model\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Hydration;
use Magento\Customer\Api\AddressRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\AddressInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderAddressInterface;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
abstract class Address extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_address';

    /** @var CustomerRepositoryInterface $customerRepository */
    protected $customerRepository;

    /** @var AddressRepositoryInterface $addressRepository */
    protected $addressRepository;

    /** @var AddressEntityFactory $addressEntityFactory */
    protected $addressEntityFactory;

    /**
     * Address constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param Config                      $config
     * @param CustomerRepositoryInterface $customerRepository
     * @param AddressRepositoryInterface  $addressRepository
     * @param AddressEntityFactory        $addressEntityFactory
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        CustomerRepositoryInterface $customerRepository,
        AddressRepositoryInterface $addressRepository,
        AddressEntityFactory $addressEntityFactory
    ) {
        parent::__construct($logger, $resourceConnection, $config);

        $this->customerRepository = $customerRepository;
        $this->addressRepository = $addressRepository;
        $this->addressEntityFactory = $addressEntityFactory;
    }

    /**
     * Get customer's last order addresses or default addresses if there is no order
     *
     * @param int            $customerId
     * @param OrderInterface $lastOrder
     * @param string         $type
     *
     * @return AddressEntity
     */
    protected function getAddressEntity($customerId, $lastOrder, $type = 'billing')
    {
        $result = null;
        $addressObject = null;

        if ($lastOrder !== false) {
            if ($type === 'billing') {
                $addressObject = $lastOrder->getBillingAddress();
            } else {
                $addressObject = $lastOrder->getShippingAddress();
            }
        } else {
            try {
                $customer = $this->customerRepository->getById($customerId);
            } catch (Exception $e) {
                $customer = null;
            }

            if ($customer !== null) {
                if ($type === 'billing') {
                    $addressId = $customer->getDefaultBilling();
                } else {
                    $addressId = $customer->getDefaultShipping();
                }

                if ($addressId) {
                    try {
                        $addressObject = $this->addressRepository->getById($addressId);
                    } catch (Exception $e) {
                        $addressObject = null;
                    }
                }
            }
        }

        if ($addressObject !== null) {
            $result = $this->addressEntityFactory->create();

            $result->setAlias($this->getAddressAlias($addressObject, $lastOrder, $customerId, $type))
                ->setMagentoId($addressObject->getId())
                ->setFirstname($addressObject->getFirstname())
                ->setLastname($addressObject->getLastname())
                ->setCompany($addressObject->getCompany())
                ->setStreet($addressObject->getStreet())
                ->setPostcode($addressObject->getPostcode())
                ->setCity($addressObject->getCity())
                ->setCountryId($addressObject->getCountryId())
                ->setPhoneNumber($addressObject->getTelephone());

            if ($type === 'shipping') {
                $result->setArchived((bool)$lastOrder);
            }
        }

        return $result;
    }

    /**
     * Defines the address alias
     *
     * @param AddressInterface|OrderAddressInterface $shippingAddress
     * @param OrderInterface|null                    $lastOrder
     * @param int                                    $customerId
     * @param string                                 $addressType
     *
     * @return string
     */
    protected function getAddressAlias($shippingAddress, $lastOrder, $customerId, $addressType = 'billing')
    {
        $result = '';

        if ($addressType === 'billing') {
            $result = ($lastOrder !== false ? $lastOrder->getEntityId() . '_INVOICE' : 'INVOICE');
        } elseif ($addressType === 'shipping') {
            $shippingAddressAlias = $shippingAddress->getCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE);
            if ($shippingAddressAlias !== null) {
                $shippingAddressAlias = $shippingAddressAlias->getValue();
            }

            if ($shippingAddressAlias === '.' || $shippingAddressAlias === '') {
                $shippingAddressAlias = $customerId;
            }

            if ($lastOrder->getEntityId()) {
                $shippingAddressAlias = $lastOrder->getEntityId() . '_' . $shippingAddressAlias;
            }

            $result = Text::upper(substr(Text::sanitize(Text::removeAccents($shippingAddressAlias)), 0, 20));
        }

        return $result;
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerId
     *
     * @return Job
     */
    protected function getJob($customerId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerId);
    }
}
