<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-09-28
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;
use Fastmag\Sync\Model\System\Connection\Edi;

/**
 * Class Delete
 *
 * Integration class used for deleting customers from Magento to Fastmag
 */
class Delete extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_delete';

    /**
     * @inheritDoc
     */
    public function run()
    {
        if (!$this->isEnabled()) {
            $this->logger->notice('Worker "' . $this->code . '" was called, even though it is disabled');
        } elseif ($this->jobs->count() <= 0) {
            $this->logger->notice('Worker "' . $this->code . '" was called, but without jobs to integrate');
        } else {
            foreach ($this->jobs->getItems() as $job) {
                try {
                    $this->processJob($job);
                } catch (JobException $e) {
                    $this->logger->error(
                        '[Job #' . $job->getId() . '] Error on customer with Magento ID #'
                        . $job->getContentId() . ': ' . $e->getMessage()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());

                    $this->jobRepository->save($job);
                }
            }
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        $customerId = (int)$job->getContentId();

        $jobHydratedData = $job->getHydratedData();

        if (!array_key_exists('fastmag_id', $jobHydratedData) || $jobHydratedData['fastmag_id'] === null) {
            $fastmagId = $this->getFastmagIdByEmail($jobHydratedData['email']);

            if ($fastmagId) {
                $jobHydratedData['fastmag_id'] = $fastmagId;

                $this->saveSyncCustomer($customerId, $fastmagId);
            }
        }

        $this->sendInstructionToFastmag($customerId, $jobHydratedData);
        $this->saveSyncCustomer($customerId, $jobHydratedData['fastmag_id'], true);
    }

    /**
     * Send CONSIGNE transaction for the customer deletion to Fastmag
     *
     * @param int   $customerId
     * @param array $hydratedData
     *
     * @return void
     *
     * @throws JobException
     */
    protected function sendInstructionToFastmag($customerId, $hydratedData)
    {
        $fastmagShop = $this->getFastmagShop($hydratedData['store_id']);

        $ediData = [
            utf8_decode($fastmagShop),
            utf8_decode('SUPPRESSION COMPTE CLIENT'),
            utf8_decode(
                'Bonjour,{CR}{LF}Le client ayant l\'ID Fastmag #' . $hydratedData['fastmag_id']
                . ' doit être supprimé dans Fastmag, conformément au RGPD.'
            ),
            utf8_decode($this->config->getValue(Config::XML_PATH_CONNECT_EDI_SHOP)),
            '1'
        ];

        try {
            $ediFormattedData = $this->edi->formatInline(Edi::EDI_HEADER_INSTRUCTION_INSERT, $ediData);

            $this->edi->post($ediFormattedData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to send instruction for the deletion of the customer #%1 to Fastmag through EDI: %2',
                $customerId,
                $e->getMessage()
            ));
        }
    }
}
