<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-10-15
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Model\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Model\Process\Worker\ToFastmag\Integration\Customer;
use Fastmag\Sync\Model\System\Connection\Edi;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_save';

    /** @inheritDoc */
    protected $subordinateWorkersCodes = ['tofastmag_integration_customer_address_save'];

    /**
     * @inheritDoc
     */
    public function run()
    {
        if (!$this->isEnabled()) {
            $this->logger->notice('Worker "' . $this->code . '" was called, even though it is disabled');
        } elseif ($this->jobs->count() <= 0) {
            $this->logger->notice('Worker "' . $this->code . '" was called, but without jobs to integrate');
        } else {
            foreach ($this->jobs->getItems() as $job) {
                try {
                    $this->processJob($job);
                } catch (JobException $e) {
                    $this->logger->error(
                        '[Job #' . $job->getId() . '] Error on customer with Magento ID #'
                        . $job->getContentId() . ': ' . $e->getMessage()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());

                    $this->jobRepository->save($job);
                }
            }
        }
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        if ($entity->getFastmagId() === null) {
            $fastmagId = $this->getFastmagIdByEmail($entity->getEmailAddress());

            if ($fastmagId) {
                $entity->setFastmagId($fastmagId);

                $this->saveSyncCustomer($entity->getMagentoId(), $fastmagId);
            }
        }

        $newFastmagId = $this->sendDataToFastmag($entity->getMagentoId(), $entity);
        if ($newFastmagId !== false) {
            $entity->setFastmagId($newFastmagId);

            $this->saveSyncCustomer($entity->getMagentoId(), $newFastmagId);
        }
    }

    /**
     * Send customer to Fastmag
     *
     * @param int            $customerId
     * @param CustomerEntity $customerEntity
     *
     * @return int|bool new Fastmag ID if it's an insertion, false otherwise
     *
     * @throws JobException
     */
    protected function sendDataToFastmag($customerId, $customerEntity)
    {
        $result = false;

        if ($customerEntity->getFastmagId() === null) {
            $result = $this->createFastmagCustomer($customerId, $customerEntity);
        } else {
            $this->updateFastmagCustomer($customerId, $customerEntity);
        }

        return $result;
    }

    /**
     * Create customer on Fastmag
     *
     * @param int            $customerId
     * @param CustomerEntity $customerEntity
     *
     * @return int
     *
     * @throws JobException
     */
    protected function createFastmagCustomer($customerId, $customerEntity)
    {
        $fastmagShop = $this->getFastmagShop($customerEntity->getStoreId());

        $billingAddress = $customerEntity->getBillingAddress();

        if ($billingAddress === null) {
            throw new JobException(__(
                'Unable to create customer #%1 on Fastmag: no billing address found.',
                $customerEntity->getMagentoId()
            ));
        }

        $ediData = [
            utf8_decode($fastmagShop),
            trim(Text::lower($customerEntity->getEmailAddress())),
            utf8_decode($billingAddress->getLastname()),
            utf8_decode($billingAddress->getFirstname()),
            $customerEntity->getPrefix(),
            utf8_decode($billingAddress->getStreetOne()),
            utf8_decode($billingAddress->getStreetTwo()),
            $billingAddress->getPostcode(),
            $billingAddress->getCity(),
            $billingAddress->getCountryId(),
            $billingAddress->getPhoneNumber(),
            $customerEntity->getBirthDay(),
            $customerEntity->getBirthMonth(),
            '', // Carte de fidélité
            '', // Remise
            self::DEFAULT_REFERER,
            '', // Observations
            '', // Mot de passe
            $billingAddress->getCompany(),
            '', // Code
            '', // Compte
            '', // Famille
            $customerEntity->getBirthYear(),
            $customerEntity->getGender(),
        ];

        try {
            $ediFormattedData = $this->edi->formatInline(Edi::EDI_HEADER_CUSTOMER_INSERT, $ediData);

            $result = $this->edi->post($ediFormattedData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to send data of the customer #%1 to Fastmag through EDI: %2. Data sent: %3',
                $customerId,
                $e->getMessage(),
                $ediData
            ));
        }

        $this->updateFastmagNewCustomer(
            $customerId,
            $result['id'],
            $customerEntity->hasSubscribedNewsletter(),
            $customerEntity->getCreatedAt()
        );

        return $result['id'];
    }

    /**
     * Update newley created customer to add some more data
     *
     * @param int    $magentoId
     * @param int    $fastmagId
     * @param bool   $newsletter
     * @param string $createdAt
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagNewCustomer($magentoId, $fastmagId, $newsletter, $createdAt)
    {
        $ediData = [
            'PubParEMail'  => $newsletter,
            'DateCreation' => $createdAt
        ];

        try {
            $ediFormattedData = $this->edi->formatInline(Edi::EDI_HEADER_CUSTOMER_UPDATE, $fastmagId)
                . PHP_EOL . $this->edi->formatBlock(Edi::EDI_LINE_CUSTOMER_ID, $ediData);

            $this->edi->post($ediFormattedData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to update customer #%1 on Fastmag through EDI: %2. Data sent: %3',
                $magentoId,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Update customer on Fastmag
     *
     * @param int            $magentoId
     * @param CustomerEntity $customerEntity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagCustomer($magentoId, $customerEntity)
    {
        $this->checkFastmagEmail($customerEntity->getFastmagId(), $customerEntity->getEmailAddress());

        $billingAddress = $customerEntity->getBillingAddress();

        if ($billingAddress === null) {
            throw new JobException(__(
                'Unable to create customer #%1 on Fastmag: no billing address found.',
                $customerEntity->getMagentoId()
            ));
        }

        $ediData = [
            'Sexe'           => $customerEntity->getGender(),
            'Nom'            => $billingAddress->getLastname(),
            'Prenom'         => $billingAddress->getFirstname(),
            'JourNaissance'  => $customerEntity->getBirthDay(),
            'MoisNaissance'  => $customerEntity->getBirthMonth(),
            'AnneeNaissance' => $customerEntity->getBirthYear(),
            'PubParEMail'    => $customerEntity->hasSubscribedNewsletter(),
            'DateCreation'   => $customerEntity->getCreatedAt(),
            'Societe'        => utf8_decode($billingAddress->getCompany()),
            'Adresse1'       => utf8_decode($billingAddress->getStreetOne()),
            'Adresse2'       => utf8_decode($billingAddress->getStreetTwo()),
            'CodePostal'     => $billingAddress->getPostcode(),
            'Ville'          => utf8_decode($billingAddress->getCity()),
            'Pays'           => utf8_decode($billingAddress->getCountryId()),
            'Telephone'      => $billingAddress->getPhoneNumber()
        ];

        try {
            $ediFormattedData =
                $this->edi->formatInline(Edi::EDI_HEADER_CUSTOMER_UPDATE, $customerEntity->getFastmagId())
                . PHP_EOL . $this->edi->formatBlock(Edi::EDI_LINE_CUSTOMER_ID, $ediData);

            $this->edi->post($ediFormattedData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to update customer #%1 on Fastmag through EDI: %2. Data sent: %3',
                $magentoId,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Check email of the customer on Fastmag, and change it if it's not the same as the one in Magento
     *
     * @param int    $fastmagId
     * @param string $email
     *
     * @return void
     *
     * @throws JobException
     */
    protected function checkFastmagEmail($fastmagId, $email)
    {
        try {
            $sql = 'SELECT Email AS email
                FROM clients
                WHERE Client = ' . $this->getSqlConnection()->escape($fastmagId);

            $rows = $this->getSqlConnection()->get($sql);

            if ($rows[0]['email'] !== $email) {
                $this->changeCustomerEmail($fastmagId, $email);
            }
        } catch (Exception $e) {
            throw new JobException(__(
                'Error when trying to check customer email on Fastmag. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                $fastmagId
            ));
        }
    }

    /**
     * Change customer email on Fastmag
     *
     * @param string $oldEmail
     * @param string $newEmail
     *
     * @return void
     *
     * @throws JobException
     */
    protected function changeCustomerEmail($oldEmail, $newEmail)
    {
        $ediData = [$oldEmail, $newEmail];

        try {
            $ediFormattedData = $this->edi->formatInline(Edi::EDI_HEADER_CHANGE_EMAIL_INSERT, $ediData);

            $this->edi->post($ediFormattedData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to update the email "%1" on Fastmag through EDI: %2. Data sent: %3',
                $oldEmail,
                $e->getMessage(),
                $ediData
            ));
        }
    }
}
