<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-09-22
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Fastmag to Magento
 * Called from Customer hydration worker only
 */
class Address extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_customer_address';

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var array $customersAddresses */
    protected $customersAddresses = [];

    /**
     * Hydration constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Config             $config
     * @param Proxy              $proxy
     * @param JobRepository      $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        Proxy $proxy,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $config, $proxy);

        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromFastmag();

        if (is_array($results)) {
            foreach ($results as $row) {
                if ($this->shouldAddAddress($row)) {
                    $job = $this->getJob($row['Client']);

                    if ($job !== null) {
                        $customerData = $job->getHydratedData();
                        $customerData['AddressesLivraison'] = $row;
                        $job->setHydratedData($customerData);
                    }

                    if (!array_key_exists($row['Client'], $this->customersAddresses)
                        || $row['Vente'] > $this->customersAddresses[$row['Client']]['last-order']
                    ) {
                        $this->customersAddresses[$row['Client']] = [
                            'last-order' => $row['Vente'],
                            'addresses'  => [$row]
                        ];
                    } else {
                        $this->customersAddresses[$row['Client']]['addresses'][] = $row;
                    }
                }
            }

            foreach ($this->jobs->getItems() as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $customersIds = $this->getCustomersIds();

        try {
            $sql = 'SELECT clientadresse.Client, clientadresse.AdrLivraison, ventes.Vente, Nom, Societe,
                    Adresse1, Adresse2, CodePostal, Ville, Telephone,
                    (SELECT CodeIso FROM pays WHERE pays.Pays = clientadresse.Pays) as PaysIso
                FROM clientadresse LEFT OUTER JOIN ventes
                    ON clientadresse.AdrLivraison = ventes.AdrLivraison AND clientadresse.Client = ventes.Client
                WHERE Archiver = 0 AND TRIM(clientadresse.AdrLivraison) <> \'\'
                    AND clientadresse.Client = ' . $this->getConnection()->escape($customersIds) . '
                ORDER BY clientadresse.Client ASC, ventes.Vente DESC';

            return $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException(
                'Error when hydrating customers addresses. Message: ' . $e->getMessage()
                . '. Customers IDs: ' . implode(', ', $customersIds)
            );
        }
    }

    /**
     * Returns the list of the customers IDs, if the jobs is linked to customers.
     *
     * @return int[]
     */
    protected function getCustomersIds()
    {
        return $this->jobs->getColumnValues(Job::CONTENT_ID);
    }

    /**
     * Defines if the address data given in parameters should be added to the customer addresses
     *
     * @param array $row
     *
     * @return bool
     */
    protected function shouldAddAddress($row)
    {
        $result = true;

        if (array_key_exists($row['Client'], $this->customersAddresses)
            && $this->customersAddresses[$row['Client']]['last-order'] !== null
            && $row['Vente'] < $this->customersAddresses[$row['Client']]['last-order']
        ) {
            $result = false;
        }

        return $result;
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerFastmagId
     *
     * @return Job
     */
    protected function getJob($customerFastmagId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerFastmagId);
    }
}
