<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-09-22
 ******************************************************************************/

namespace Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration\Product;

use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Process\Worker\ToMagento\Hydration\Product;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Price
 *
 * Hydration class used for inserting or updating products prices from Fastmag to Magento
 */
class Price extends Product
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_inventory';

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            $fastmagRef = $this->currentJob->getContentId();

            if ($this->getMagentoProductId($fastmagRef) !== false) {
                $this->currentJob = $job;

                try {
                    $hydratedData = $this->getDataFromFastmag();
                    $job->setHydratedData($hydratedData);
                } catch (JobException $e) {
                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            } else {
                $this->logger->notice(
                    'Can not change prices of the product with Fastmag ref "' . $fastmagRef
                    . '", as it does not exist in Magento'
                );
            }
        }
    }

    /**
     * @inheritDoc
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();
        [$fastmagParentRef, $fastmagColorRef] = $this->getRealFastmagRef();

        if ($this->isPack($fastmagParentRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $hydratedData = [
            'parent_ref' => $fastmagParentRef,
            'color_ref' => $fastmagColorRef,
            'is_color' => ($fastmagColorRef !== ''),
            'magento_id' => $this->getMagentoProductId($fastmagRef)
        ];

        $hydratedData = array_merge($hydratedData, $this->getBasicData($fastmagParentRef));

        if ($this->productMustBeSynced($hydratedData)) {
            $storesList = $this->storeRepository->getList();

            foreach ($storesList as $store) {
                try {
                    $this->currentRule = $this->storesellerRuleRepository->getByStoreId($store->getId());
                } catch (Exception $e) {
                    continue;
                }

                $fastmagStaticData = $this->getStaticData($fastmagParentRef, $fastmagColorRef);

                $hydratedDataStore['children'] = $fastmagStaticData;

                $hydratedData[$store->getCode()] = $hydratedDataStore;
            }
        }

        return $hydratedData;
    }

    /**
     * @inheritDoc
     */
    protected function getStaticData($fastmagParentRef, $fastmagColorRef)
    {
        $result = [];

        try {
            $rateCode = $this->currentRule->getRateCode();
            $fastmagShop = $this->currentRule->getFastmagShop();

            $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagParentRef, $fastmagShop, $rateCode);
            $combinationPriceSubquery = $this->getCombinationPriceSubquery($fastmagParentRef, $rateCode);
            $priceRateSubquery = $this->getPriceRateSubquery($fastmagParentRef, $rateCode);
            $salesPriceSubquery = $this->getSalesPriceSubquery($fastmagParentRef, $rateCode);
            $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
            $buyingPriceSubquery = $this->getBuyingPriceSubquery($fastmagParentRef);

            $onlyStoresellerCombination = '';
            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $onlyStoresellerCombination = 'HAVING enable_stock = 1 ';
            }

            $colorCondition = '';
            if ($fastmagColorRef !== '') {
                $colorCondition = ' AND stock.Couleur = ' . $this->getConnection()->escape($fastmagColorRef);
            }

            $sql = 'SELECT
                    (' . $standardPriceSubquery . ') AS standard_price,
                    (' . $combinationPriceSubquery . ') AS combination_price,
                    (' . $priceRateSubquery . ') AS price_rate,
                    (' . $salesPriceSubquery . ') AS sales_price,
                    (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                    (' . $buyingPriceSubquery . ') AS buying_price,
                    stock.Taille AS size,
                    stock.Couleur AS color
                FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
                WHERE stock.AR = 1 AND stock.BarCode = ' . $this->getConnection()->escape($fastmagParentRef, true) . '
                    ' . $colorCondition . '
                GROUP BY stock.Barcode, stock.Taille, stock.Couleur
                ' . $onlyStoresellerCombination . '
                ORDER BY stock.Couleur, stock.Taille';

            $rows = $this->getConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        foreach ($rows as $row) {
            if (!array_key_exists($row['color'], $result)) {
                $result[$row['color']] = [$row['size'] => $row];
            } else {
                $result[$row['color']][$row['size']] = $row;
            }
        }

        return $result;
    }
}
