<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-10-21
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\System\Connection\Edi\Config;
use Magento\Framework\HTTP\Client\Curl;

/**
 * Class Edi
 *
 * Config class to connect to EDI
 */
class Edi
{
    /** @var string */
    public const EDI_HEADER_CUSTOMER_INSERT = 'CLIENT';

    /** @var string */
    public const EDI_HEADER_CHANGE_EMAIL_INSERT = 'CHANGEEMAIL';

    /** @var string */
    public const EDI_HEADER_CUSTOMER_UPDATE = 'CLIENTUPDATE';

    /** @var string */
    public const EDI_LINE_CUSTOMER_ID = 'LIGNECLIENT';

    /** @var string */
    public const EDI_LINE_ADDRESS_ID = 'ADRLIV';

    /** @var string */
    public const EDI_HEADER_INSTRUCTION_INSERT = 'CONSIGNE';

    /** @var string */
    public const EDI_HEADER_TRANSACTION_INSERT = 'ENTETE';

    /** @var string */
    public const EDI_HEADER_TRANSACTION_DEFAULT_VALUE_ORIGIN = 'VSHOP';

    /** @var string */
    public const EDI_LINE_TRANSACTION_PRODUCT_ID = 'LIGNE';

    /** @var string */
    public const EDI_FOOTER_TRANSACTION_INSERT = 'REGLEMENT';

    /** @var string */
    public const EDI_HEADER_CANCELLATION_INSERT = 'ANNULATION';

    /** @var Config $config */
    protected $config;

    /** @var Curl $curlClient */
    protected $curlClient;

    /** @var Logger $logger */
    protected $logger;

    /** @var string $endpoint */
    protected $endpoint = 'ediwebsrv.ips';

    /**
     * Edi constructor
     *
     * @param Config $config
     * @param Curl   $curlClient
     * @param Logger $logger
     */
    public function __construct(Config $config, Curl $curlClient, Logger $logger)
    {
        $this->config = $config;
        $this->curlClient = $curlClient;
        $this->logger = $logger;
    }

    /**
     * Get endpoint
     *
     * @return string
     */
    public function getEndpoint()
    {
        return $this->endpoint;
    }

    /**
     * Try connect to EDI
     *
     * @return bool
     *
     * @throws NoConnectionException
     *
     * @todo find a better way to test the connection
     */
    public function connect()
    {
        $result = null;
        $config = $this->config->getFormattedConfig();
        $url = $this->getUrl($config);

        $curlOptions = [
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true
        ];

        $this->postCurl($url, 'CONSIGNE', [], $curlOptions);
        $response = $this->getResponseBodyCurl();

        if (!$this->isConnectResponseOK($response)) {
            $this->logger->critical(__(
                'Failed to connect to EDI: Response: %1 - Please verify your settings. URL: %2. Config used: %3',
                $response,
                $url,
                implode($config)
            ));

            throw new NoConnectionException($response);
        }

        $result = true;

        return $result;
    }

    /**
     * Post data to EDI
     *
     * @param string $ediData
     * @param array  $params
     *
     * @return array
     *
     * @throws NoConnectionException
     */
    public function post($ediData, $params = [])
    {
        $curlOptions = [
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CONNECTTIMEOUT => 60,
            CURLOPT_TIMEOUT        => 90,
            CURLOPT_FRESH_CONNECT  => true
        ];

        $time = $this->postCurl($this->getUrl(), $ediData, $params, $curlOptions);

        $response = $this->getResponseBodyCurl();

        $status = 'KO';
        $fastmagId = $fastmagIdBis = false;
        $ediError = false;

        if (!$this->isConnectResponseOK($response)) {
            $this->logger->critical(
                __('Failed to connect to EDI: Response: %1 - Please verify your settings', $response)
            );

            throw new NoConnectionException($response);
        }

        if (preg_match_all('/OK\|(.*)/i', $response, $matches)) {
            $response = explode('|', $matches[0][0]);
            $status = 'OK';
            $fastmagId = trim(trim($response[1], "\r"));

            if (isset($response[2])) {
                $fastmagIdBis = (int)trim(trim($response[2], "\r"));
            }
        } elseif (preg_match_all('/KO\|(.*)/i', $response, $matches)) {
            if (strpos($ediData, 'ANNULATION') !== false && strpos($matches[0][0], 'DEJA ANNULE') !== false) {
                $response = explode('|', $matches[0][0]);
                $status = 'OK';
                $ediDataArray = explode('|', $ediData);
                $fastmagId = $ediDataArray[3];
                $ediError = implode('-', $response);
            } else {
                $response = explode('|', $matches[0][0]);
                $ediError = implode('-', $response);

                $this->logger->critical(
                    __('Failed to send data to EDI: Response: %1', $ediError)
                );

                throw new NoConnectionException($response);
            }
        } else {
            // Other case nor OK or KO
            $httpCode = $this->getResponseStatusCurl();
            $ediError = 'No response - case nor OK or KO : ' . $response . ' : ' . $httpCode . ' - in ' . $time . ' s';

            if (preg_match_all('|<h[^>]+>(.*)</h[^>]+>|iU', $response, $headings)) {
                $ediError = 'KO - ' . $headings[1][0];
            }

            $this->logger->critical(
                __('Failed to send data to EDI: Response: %1', $ediError)
            );

            throw new NoConnectionException($response);
        }

        return [
            'status'    => $status,
            'id'        => $fastmagId,
            'id_bis'    => $fastmagIdBis,
            'post_data' => $ediData,
            'response'  => $response,
            'error'     => utf8_encode($ediError)
        ];
    }

    /**
     * Internal method to send data to EDI through CURL
     *
     * @param string $url
     * @param string $data
     * @param array  $params
     * @param array  $curlOptions
     *
     * @return float
     */
    protected function postCurl($url, $data, $params = [], $curlOptions = [])
    {
        $start = microtime(true);

        $this->curlClient->setOptions($curlOptions);

        $this->curlClient->post(
            $url,
            [
                'compte'   => (array_key_exists('login', $params) ? $params['user'] : $this->config->getUser()),
                'motpasse' => (array_key_exists('password', $params)
                    ? $params['password'] : $this->config->getPassword()),
                'enseigne' => (array_key_exists('chain', $params) ? $params['chain'] : $this->config->getChain()),
                'magasin'  => (array_key_exists('shop', $params) ? $params['shop'] : $this->config->getShop()),
                'data'     => $data
            ]
        );

        $end = microtime(true);

        return $end - $start;
    }

    /**
     * Returns last response body
     *
     * @return string
     */
    protected function getResponseBodyCurl()
    {
        return $this->curlClient->getBody();
    }

    /**
     * Returns last response headers
     *
     * @return array
     */
    protected function getResponseHeadersCurl()
    {
        return $this->curlClient->getHeaders();
    }

    /**
     * Returns last response status
     *
     * @return int
     */
    protected function getResponseStatusCurl()
    {
        return $this->curlClient->getStatus();
    }

    /**
     * Get EDI URL
     *
     * @param array|null $config
     * @param array|null $params
     *
     * @return string|null
     */
    public function getUrl($config = null, $params = null)
    {
        if ($config === null) {
            $config = $this->config->getFormattedConfig();
        }
        if (!$this->config->isConfigReadyForUrl($config)) {
            return null;
        }

        $result = 'http' . ($config['ssl'] ? 's' : '') . '://'
            . $config['host'] . ($config['port'] ? ':' . $config['port'] : '')
            . '/' . $this->getEndpoint();

        if ($params !== null) {
            $result .= '?' . http_build_query($params, null, '&');
        }

        return $result;
    }

    /**
     * Check if the connection response is OK
     *
     * @param string|null $response
     *
     * @return bool
     */
    protected function isConnectResponseOK($response)
    {
        $result = false;
        $errorMessages = [
            'MAGASIN INCONNU',
            'CONNEXION REFUSEE',
            'Unknown database',
            'Les informations fournies ne permettent pas votre identification'
        ];
        $errorFound = false;

        if (!empty($response)) {
            foreach ($errorMessages as $message) {
                if (strpos($response, $message) !== false) {
                    $errorFound = true;
                }
            }

            if (!$errorFound) {
                $result = true;
            }
        }

        return $result;
    }

    /**
     * Format data for "inline mode"
     *
     * @param string $header
     * @param mixed  $data
     *
     * @return string
     */
    public function formatInline($header, $data)
    {
        if (!is_array($data)) {
            $data = [$data];
        }

        return implode('|', array_unshift($data, $header));
    }

    /**
     * Format data for "block mode"
     *
     * @param string $lineId
     * @param array  $data
     *
     * @return string
     */
    public function formatBlock($lineId, $data)
    {
        $ediData = '';

        foreach ($data as $field => $value) {
            $ediData .= $lineId . '|' . $field . '|' . $value . '|' . PHP_EOL;
        }

        return trim($ediData, PHP_EOL);
    }
}
