<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2020 HOMEMADE.IO SAS
 * @date      2020-10-01
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection\Sql;

use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\FastmagSetup\Installer;
use Fastmag\Sync\FastmagSetup\Uninstaller;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config\Provider\DirectSql as DirectSqlConfig;
use Fastmag\Sync\Model\Config\Provider\Edi;
use Magento\Framework\DB\Adapter\Pdo\Mysql as MysqlAdapter;
use Magento\Framework\DB\Adapter\Pdo\MysqlFactory as MysqlAdapterFactory;
use Magento\Framework\Model\ResourceModel\Type\Db\Pdo\Mysql as MysqlResource;
use PDO;

/**
 * Class DirectSql
 *
 * Class to connect and query to Fastmag through direct SQL interfaces
 */
class DirectSql extends MysqlResource implements SqlInterface
{
    /* @var MysqlAdapter */
    protected $connection;

    /** @var Edi */
    protected $ediConfigProvider;

    /** @var DirectSqlConfig */
    protected $directSqlConfigProvider;

    /* @var MysqlAdapterFactory */
    protected $connectionFactory;

    /** @var Logger */
    protected $logger;

    /** @var Installer $installer */
    protected $installer;

    /** @var Uninstaller $uninstaller */
    protected $uninstaller;

    /**
     * DirectSql constructor.
     *
     * @param DirectSqlConfig          $directSqlConfigProvider
     * @param Edi                      $ediConfigProvider
     * @param Logger                   $logger
     * @param Installer                $installer
     * @param Uninstaller              $uninstaller
     * @param MysqlAdapterFactory|null $mysqlFactory
     */
    public function __construct(
        DirectSqlConfig $directSqlConfigProvider,
        Edi $ediConfigProvider,
        Logger $logger,
        Installer $installer,
        Uninstaller $uninstaller,
        MysqlAdapterFactory $mysqlFactory = null
    ) {
        $this->directSqlConfigProvider = $directSqlConfigProvider;
        $this->ediConfigProvider = $ediConfigProvider;
        $this->logger = $logger;
        $this->installer = $installer;
        $this->uninstaller = $uninstaller;

        $config = $this->getConfigFromProvider();

        parent::__construct($config, $mysqlFactory);
    }

    /**
     * @inheritDoc
     */
    public function connect()
    {
        if ($this->connection === null) {
            $this->connection = $this->getConnection();
        }

        return $this->connection !== null;
    }

    /**
     * @inheritDoc
     */
    public function get($sql)
    {
        try {
            $result = $this->connection->query($sql);
        } catch (Exception $e) {
            throw new NoConnectionException($e);
        }

        return $result === null ? [] : $result->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * @inheritDoc
     */
    public function post($sql)
    {
        $result = true;

        try {
            $result = $this->connection->query($sql) !== null;
        } catch (Exception $e) {
            throw new NoConnectionException($e);
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function multiPost($sql)
    {
        $result = true;

        try {
            $result = $this->connection->multiQuery($sql) !== null;
        } catch (Exception $e) {
            throw new NoConnectionException($e);
        }

        return $result;
    }

    /**
     * Get Direct SQL config stored in conf
     *
     * @return array
     */
    protected function getConfigFromProvider()
    {
        return [
            'host'     => $this->directSqlConfigProvider->getHost() . ':' . $this->directSqlConfigProvider->getPort(),
            'username' => $this->directSqlConfigProvider->getUser(),
            'password' => $this->directSqlConfigProvider->getPassword(),
            'dbname'   => $this->getDatabase()
        ];
    }

    /**
     * Get database name
     *
     * @return string
     */
    public function getDatabase()
    {
        return 'fastmag_' . strtolower($this->ediConfigProvider->getChain());
    }

    /**
     * @inheritDoc
     */
    public function escape($value, $forceString = false)
    {
        if (is_array($value)) {
            $result = '';

            foreach ($value as $row) {
                $result .= $this->escape($row, $forceString) . ', ';
            }

            $result = trim($result, ', ');
        } else {
            $result = stripslashes($value);

            if ($forceString || !is_numeric($result)) {
                $search = ['\\', "\0", "\n", "\r", "\x1a", "'", '"'];
                $replace = ['\\\\', '\\0', '\\n', '\\r', "\Z", "\'", '\"'];

                $result = '\'' . strip_tags(nl2br(str_replace($search, $replace, $result))) . '\'';
            }
        }

        return $result;
    }

    /**
     * Reinstall triggers on Fastmag using SQL proxy
     *
     * @throws Exception
     */
    public function reinstallTriggers()
    {
        $sql = $this->uninstaller->getUninstallQueries();

        foreach ($sql as $entityType) {
            foreach ($entityType as $query) {
                $this->post($query);
            }
        }

        $sql = $this->installer->getInstallQueries();

        $sql['triggers'] = $this->filterTriggersByConfig($sql['triggers']);

        foreach ($sql as $entityType) {
            foreach ($entityType as $query) {
                $this->multiPost($query);
            }
        }
    }

    /**
     * Filter the triggers list to install given the configuration of the Magento instance
     *
     * @param array $queriesList
     *
     * @return array
     *
     * @todo to fill when the configuration fields will be plugged
     */
    protected function filterTriggersByConfig($queriesList)
    {
        return $queriesList;
    }
}
