<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Controller\Adminhtml\Jobqueue\Tomagento;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterfaceFactory;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoSearchResultsInterface;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface;
use Fastmag\Sync\Logger\Logger;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Backend\Model\View\Result\Redirect;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Action\HttpGetActionInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class ResetInventoryLevels
 *
 * ResetInventoryLevels controller
 */
class ResetInventoryLevels extends Action implements HttpGetActionInterface
{
    /** @var ToMagentoInterfaceFactory $jobFactory */
    protected $jobFactory;

    /** @var ToMagentoRepositoryInterface */
    protected $jobRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var Logger $logger */
    protected $logger;

    /**
     * ResetPrices constructor.
     *
     * @param Context                      $context
     * @param ToMagentoInterfaceFactory    $jobFactory
     * @param ToMagentoRepositoryInterface $jobRepository
     * @param SearchCriteriaBuilder        $searchCriteriaBuilder
     * @param Logger                       $logger
     */
    public function __construct(
        Context $context,
        ToMagentoInterfaceFactory $jobFactory,
        ToMagentoRepositoryInterface $jobRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        Logger $logger
    ) {
        parent::__construct($context);

        $this->jobFactory = $jobFactory;
        $this->jobRepository = $jobRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->logger = $logger;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToMagentoInterface::JOB_CODE, 'tomagento_integration_product_reset_inventory')
            ->addFilter(ToMagentoInterface::CONTENT_ID, 'all')
            ->create();

        $resetJobs = $this->jobRepository->getList($searchCriteria);

        if ($resetJobs->getTotalCount() === 1) {
            $items = $resetJobs->getItems();
            $job = reset($items);

            $job->setStatus('1')
                ->setHydratedData(null)
                ->setProcessedAt(null);
        } else {
            if ($resetJobs->getTotalCount() > 1) {
                $this->deleteOldResetJobs($resetJobs);
            }

            $job = $this->jobFactory->create();

            $job->setContentId('all')
                ->setJobCode('tomagento_integration_product_reset_inventory')
                ->setPriority('1000');
        }

        try {
            $this->jobRepository->save($job);
        } catch (CouldNotSaveException $e) {
            $this->messageManager->addErrorMessage(
                __('Unable to create the job to update the catalog\'s inventory levels. Please check your logs.')
            );

            $this->logger->critical(
                'Can not save job ' . $job->getJobCode() . ' to reset all catalog\'s prices: '
                . $e->getMessage() . "\n" . $e->getTraceAsString()
            );
        }

        $this->messageManager->addSuccessMessage(
            __('Your catalog\'s inventory levels will be updated in a few minutes.')
        );

        /** @var Redirect $resultRedirect */
        $resultRedirect = $this->resultFactory->create(ResultFactory::TYPE_REDIRECT);

        return $resultRedirect->setPath('*/*/');
    }

    /**
     * Delete old reset jobs if there are more than one
     *
     * @param ToMagentoSearchResultsInterface $resetJobs
     *
     * @return void
     */
    protected function deleteOldResetJobs($resetJobs)
    {
        try {
            foreach ($resetJobs as $resetJob) {
                $this->jobRepository->delete($resetJob);
            }
        } catch (CouldNotDeleteException $e) {
            $this->messageManager->addErrorMessage(
                __('There is already too many reset jobs in the queue and they can\'t be deleted.')
            );
        }
    }
}
