<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process;

use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\Collection;

/**
 * Class Manager
 *
 * Parent class of the process manager (remote sync, to magento and to fastmag)
 */
class Manager
{
    /** @var Logger $logger */
    protected $logger;

    /** @var Config $config */
    protected $config;

    /** @var WorkerFactory $workerFactory */
    protected $workerFactory;

    /**
     * Job queue manager constructor.
     *
     * @param Logger        $logger
     * @param Config        $config
     * @param WorkerFactory $workerFactory
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory
    ) {
        $this->logger = $logger;
        $this->config = $config;
        $this->workerFactory = $workerFactory;
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites()
    {
        $result = true;

        if (!$this->isSyncEnabled()) {
            throw new ProcessException('Synchronization is currently disabled. Remote sync can not be done.');
        }

        if ($this->getWebsiteId() === null) {
            throw new ProcessException('Website ID not set in config.');
        }

        if ($this->getSyncLimit() <= 0) {
            throw new ProcessException('Process queue limit size not set in config.');
        }

        return $result;
    }

    /**
     * Check if global synchronization is enabled
     *
     * @return bool
     */
    protected function isSyncEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_JOBQUEUE_SYNC_ENABLED);
    }

    /**
     * Check if website ID for synchonization is set in config
     *
     * @return string
     */
    protected function getWebsiteId()
    {
        return $this->config->getValue(Config::XML_PATH_JOBQUEUE_SYNC_WEBSITE_ID);
    }

    /**
     * Get sync limit
     *
     * @return int
     */
    protected function getSyncLimit()
    {
        return (int)$this->config->getValue(Config::XML_PATH_JOBQUEUE_ADVANCED_JOBS_LIMIT);
    }

    /**
     * Run worker and subordinate workers if the worker has any
     *
     * @param Worker     $worker
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function runWorker($worker, $jobsCollection)
    {
        $worker->setJobs($jobsCollection)->run();

        if ($worker->hasSubordinateWorkers()) {
            foreach ($worker->getSubordinateWorkersCodes() as $subWorker) {
                $subWorker = $this->workerFactory->create($subWorker);
                $subWorker->setJobs($jobsCollection)->run();
            }
        }
    }
}
