<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-04-06
 ******************************************************************************/

namespace Fastmag\Sync\Process\Manager;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Job;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository;
use Fastmag\Sync\Model\ResourceModel\Jobqueue\ToMagento\Collection;
use Fastmag\Sync\Process\Manager;
use Fastmag\Sync\Process\Worker\ToMagento\Clean;
use Fastmag\Sync\Process\Worker\ToMagento\Integration;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product\Inventory;
use Fastmag\Sync\Process\Worker\ToMagento\Scheduled;
use Fastmag\Sync\Process\WorkerFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\Area;
use Magento\Framework\App\State;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class ToMagento
 *
 * Workers manager for synchronization from Fastmag to Magento
 */
class ToMagento extends Manager
{
    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var ToMagentoRepository $jobRepository */
    protected $jobRepository;

    /** @var State $state */
    protected $state;

    /**
     * ToMagento constructor
     *
     * @param Logger                $logger
     * @param Config                $config
     * @param WorkerFactory         $workerFactory
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param SortOrderBuilder      $sortOrderBuilder
     * @param ToMagentoRepository   $jobRepository
     * @param State                 $state
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        ToMagentoRepository $jobRepository,
        State $state
    ) {
        parent::__construct($logger, $config, $workerFactory);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->jobRepository = $jobRepository;
        $this->state = $state;
    }

    /**
     * Main method
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return void
     *
     * @throws ProcessException
     */
    public function run($jobCode = null, $jobId = null)
    {
        try {
            $checkOk = $this->checkPrerequisites($jobCode, $jobId);

            if ($checkOk) {
                $this->logger->debug('To Magento Sync - Begin process');

                $this->state->setAreaCode(Area::AREA_ADMINHTML);

                $this->moveScheduledJobs();

                $jobsCollectionsByPriorityCode = $this->getJobsCollections($jobCode, $jobId);

                if (count($jobsCollectionsByPriorityCode) > 0) {
                    foreach ($jobsCollectionsByPriorityCode as $collectionPriority => $jobsCollectionByPriorityCode) {
                        /** @var Collection $jobsCollection */
                        foreach ($jobsCollectionByPriorityCode as $collectionJobCode => $jobsCollection) {
                            $this->processJobs($collectionJobCode, $jobsCollection);
                        }
                    }
                }

                /** @var Clean cleanWorker */
                $cleanWorker = $this->workerFactory->create('tomagento_clean');
                $cleanWorker->run();
            }
        } catch (ProcessException $e) {
            $this->logger->critical($e->getMessage());

            throw $e;
        } catch (LocalizedException $e) {
            $this->logger->critical($e->getMessage());

            throw new ProcessException($e->getMessage());
        }
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return bool
     *
     * @throws ProcessException
     */
    protected function checkPrerequisites($jobCode = null, $jobId = null)
    {
        $result = parent::checkPrerequisites();

        if ($result) {
            if ($jobId !== null) {
                try {
                    $job = $this->jobRepository->getById($jobId);
                    $jobCode = $job->getJobCode();
                } catch (Exception $e) {
                    throw new ProcessException('Job ID #' . $jobId . ' does not exist in queue.');
                }
            }

            if ($jobCode !== null) {
                /** @var Integration $worker */
                $worker = $this->workerFactory->create($jobCode);

                if (!$worker->isEnabled()) {
                    throw new ProcessException('Jobs "' . $jobCode . '" are not enabled.');
                }
            }
        }

        return $result;
    }

    /**
     * Run the specific worker to move scheduled jobs into instant queue
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function moveScheduledJobs()
    {
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_PRICE_MANAGE_TIER_PRICES)) {
            /** @var Scheduled $scheduledWorker */
            $scheduledWorker = $this->workerFactory->create('tomagento_move_scheduled');
            $scheduledWorker->run();
        }
    }

    /**
     * Get array of jobs collection to process
     *
     * @param string|null $jobCode
     * @param string|null $jobId
     *
     * @return array
     */
    protected function getJobsCollections($jobCode, $jobId)
    {
        $sortOrderPriority = $this->sortOrderBuilder->setField(ToMagentoInterface::PRIORITY)
            ->setDescendingDirection()
            ->create();
        $sortOrderJobId = $this->sortOrderBuilder->setField(ToMagentoInterface::JOB_ID)
            ->setAscendingDirection()
            ->create();

        $this->searchCriteriaBuilder->addFilter(ToMagentoInterface::STATUS, ToMagentoInterface::STATUS_OK)
            ->addFilter(ToMagentoInterface::PROCESSED_AT, null, 'null')
            ->setSortOrders([$sortOrderPriority, $sortOrderJobId])
            ->setPageSize($this->getSyncLimit());

        if ($jobId) {
            $this->searchCriteriaBuilder->addFilter(ToMagentoInterface::JOB_ID, $jobId);
        } elseif ($jobCode) {
            $this->searchCriteriaBuilder->addFilter(ToMagentoInterface::JOB_CODE, $jobCode);
        }

        $searchCriteria = $this->searchCriteriaBuilder->create();

        return $this->jobRepository->getListByPriorityAndCode($searchCriteria);
    }

    /**
     * Process jobs collection
     *
     * @param string     $jobCode
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function processJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $worker = $this->getJobsWorker($jobCode);

            if ($worker->isEnabled()) {
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Beginning hydration');

                $this->hydrateJobs($jobCode, $jobsCollection);
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Hydration done, beginning integration.');

                $this->runWorker($worker, $jobsCollection);
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Integration done, beginning results handling');

                $this->handleResults($worker);
                $this->logger->debug('Jobs ' . $worker->getCode() . ' - Results handling done');
            } else {
                $this->logger->debug('Jobs "' . $worker->getCode() . '" are not enabled.');
            }
        }
    }

    /**
     * Return worker instance for the current job code
     *
     * @param string $jobCode
     *
     * @return Integration
     */
    protected function getJobsWorker($jobCode)
    {
        if (strpos($jobCode, 'tomagento_integration_product_') === 0
            && $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_INVENTORY)
        ) {
            $this->logger->debug(
                'Jobs ' . $jobCode . ' - "Stock only" config field set, '
                . 'switched to tomagento_integration_product_quantity worker'
            );

            /** @var Inventory $result */
            $result = $this->workerFactory->create('tomagento_integration_product_quantity');
        } else {
            /** @var Integration $result */
            $result = $this->workerFactory->create($jobCode);
        }

        return $result;
    }

    /**
     * Run hydration workers to hydrate current jobs and add entities created to the jobs collection
     *
     * @param string     $jobCode
     * @param Collection $jobsCollection
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function hydrateJobs($jobCode, $jobsCollection)
    {
        if ($jobsCollection->count() > 0) {
            $hydrationJobCode = $this->getHydrationJobCode($jobCode);

            if ($hydrationJobCode) {
                /** @var Hydration $worker */
                $worker = $this->workerFactory->create($hydrationJobCode);

                /** @throws ProcessException */
                $this->runWorker($worker, $jobsCollection);
            }
        }
    }

    /**
     * Get hydration worker code for sync job code
     *
     * @todo make this method more generic
     *
     * @param string $jobCode
     *
     * @return string
     */
    protected function getHydrationJobCode($jobCode)
    {
        $result = '';

        if (stripos($jobCode, 'product') !== false) {
            if (stripos($jobCode, 'reset') !== false) {
                if (stripos($jobCode, 'inventory') !== false) {
                    $result = 'tomagento_hydration_product_reset_inventory';
                } elseif (stripos($jobCode, 'price') !== false) {
                    $result = 'tomagento_hydration_product_reset_price';
                }
            } else {
                if (stripos($jobCode, 'quantity') !== false) {
                    $result = 'tomagento_hydration_product_inventory';
                } elseif (stripos($jobCode, 'price') !== false) {
                    $result = 'tomagento_hydration_product_price';
                } else {
                    $result = 'tomagento_hydration_product';
                }
            }
        } elseif (stripos($jobCode, 'customer') !== false) {
            $result = 'tomagento_hydration_customer';
        } elseif (stripos($jobCode, 'address') !== false) {
            $result = 'tomagento_hydration_address';
        } elseif (stripos($jobCode, 'order') !== false) {
            $result = 'tomagento_hydration_order';
        }

        return $result;
    }

    /**
     * Handle jobs results
     *
     * @param Integration $worker
     */
    protected function handleResults($worker)
    {
        $errors = $worker->getErrors();

        /** @var Job $job */
        foreach ($worker->getJobs() as $job) {
            if ($worker->isJobInError($job)) {
                $e = new Exception();
                $job->setStatus(Job::STATUS_ERROR)
                    ->setMessage($errors[$job->getId()])
                    ->setTrace($e->getTraceAsString());

                $this->logger->error(
                    'Error on job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId(). ': '
                    . $job->getMessage()
                );
            } else {
                $job->setStatus(Job::STATUS_OK)
                    ->setProcessedAt(date('Y-m-d H:i:s'));
            }

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $e) {
                $this->logger->critical(
                    'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                    . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                );
            }
        }
    }
}
