<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Address as AbstractAddress;
use Magento\Customer\Api\AddressRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating customers addresses from Magento to Fastmag
 */
class Address extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_address';

    /** @var OrderRepositoryInterface $orderRepository */
    protected $orderRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * Address constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param Config                      $config
     * @param CustomerRepositoryInterface $customerRepository
     * @param AddressRepositoryInterface  $addressRepository
     * @param AddressEntityFactory        $addressEntityFactory
     * @param OrderRepositoryInterface    $orderRepository
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     * @param SortOrderBuilder            $sortOrderBuilder
     * @param JobRepository               $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        CustomerRepositoryInterface $customerRepository,
        AddressRepositoryInterface $addressRepository,
        AddressEntityFactory $addressEntityFactory,
        OrderRepositoryInterface $orderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $config,
            $customerRepository,
            $addressRepository,
            $addressEntityFactory
        );

        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($results as $customerId => $row) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    /** @var CustomerEntity $customerEntity */
                    $customerEntity = $job->getEntity();
                    $customerEntity->setBillingAddress($row['billing_address'])
                        ->setShippingAddress($row['shipping_address']);

                    $job->setHydratedData($customerEntity->export())
                        ->setEntity($customerEntity);
                }
            }

            foreach ($this->jobs->getItems() as $job) {
                try {
                    if ($job->getMessage() === null) {
                        $this->jobRepository->save($job);
                    }
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return AddressEntity[]
     */
    protected function getDataFromMagento()
    {
        $customersIds = $this->getCustomersIds();
        $result = [];

        foreach ($customersIds as $customerId) {
            /** @var Order $lastOrder */
            $lastOrder = $this->getLastOrder($customerId);

            $billingAddress = $this->getAddressEntity($customerId, $lastOrder, 'billing');
            $shippingAddress = $this->getAddressEntity($customerId, $lastOrder, 'shipping');

            $hydratedData = [
                'billing_address'  => $billingAddress,
                'shipping_address' => $shippingAddress
            ];

            $result[$customerId] = $hydratedData;
        }

        return $result;
    }

    /**
     * Returns the list of the customers IDs, if the jobs is linked to customers.
     *
     * @return int[]
     */
    protected function getCustomersIds()
    {
        return $this->jobs->getColumnValues(Job::CONTENT_ID);
    }

    /**
     * Get last order ID made by the customer (if the customer has made one)
     *
     * @param int $customerId
     *
     * @return OrderInterface|bool
     */
    protected function getLastOrder($customerId)
    {
        $result = false;

        $sortOrder = $this->sortOrderBuilder->setField(OrderInterface::CREATED_AT)
            ->setDescendingDirection()
            ->create();

        $searchCriteria = $this->searchCriteriaBuilder->addFilter(OrderInterface::CUSTOMER_ID, $customerId)
            ->addSortOrder($sortOrder)
            ->setPageSize(1)
            ->create();

        $orderList = $this->orderRepository->getList($searchCriteria);

        if ($orderList->getTotalCount() === 1) {
            $result = $orderList->getItems()[0];
        }

        return $result;
    }
}
