<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Delete
 *
 * Hydration class used for deleting customers from Magento to Fastmag
 */
class Delete extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_customer_delete';

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromMagento();

        if (is_array($results)) {
            foreach ($results as $customerId => $row) {
                $job = $this->getJob($customerId);

                if ($job !== null) {
                    $job->setHydratedData($row);
                }
            }

            foreach ($this->jobs->getItems() as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );

                    $job->setMessage($e->getMessage())
                        ->setTrace($e->getTraceAsString());
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $this->entities = [];
        $customersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);

        foreach ($customersIds as $customerId) {
            $customerEntity = $this->customerEntityFactory->create();
            $customerEntity->setMagentoId($customerId);
            $this->entities[$customerId] = $customerEntity;
        }

        $this->getSyncedCustomers($customersIds);
        $this->getMagentoCustomersData($customersIds);

        return $this->entities;
    }

    /**
     * Get Magento data for the current jobs
     *
     * @param int[] $customersIds
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getMagentoCustomersData($customersIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $customersIds, 'in')
            ->create();

        try {
            $customersList = $this->customerRepository->getList($searchCriteria);

            foreach ($customersList->getItems() as $customer) {
                $currentEntity = $this->entities[$customer->getId()];

                $currentEntity->setEmailAddress($customer->getEmail())
                    ->setStoreId($customer->getStoreId());
            }
        } catch (Exception $e) {
            throw new ProcessException(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                implode(', ', $customersIds)
            );
        }
    }
}
