<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\AddressFactory as AddressEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Address as AbstractAddress;
use Magento\Customer\Api\AddressRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SortOrderBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Api\OrderRepositoryInterface;

/**
 * Class Address
 *
 * Hydration class used for inserting or updating orders delivery addresses from Magento to Fastmag
 */
class Address extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_address';

    /** @var OrderRepositoryInterface $orderRepository */
    protected $orderRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var SortOrderBuilder $sortOrderBuilder */
    protected $sortOrderBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /**
     * Address constructor.
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param Config                      $config
     * @param CustomerRepositoryInterface $customerRepository
     * @param AddressRepositoryInterface  $addressRepository
     * @param AddressEntityFactory        $addressEntityFactory
     * @param OrderRepositoryInterface    $orderRepository
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     * @param SortOrderBuilder            $sortOrderBuilder
     * @param JobRepository               $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        OrderRepositoryInterface $orderRepository,
        CustomerRepositoryInterface $customerRepository,
        AddressRepositoryInterface $addressRepository,
        AddressEntityFactory $addressEntityFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        SortOrderBuilder $sortOrderBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $config,
            $customerRepository,
            $addressRepository,
            $addressEntityFactory
        );

        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->sortOrderBuilder = $sortOrderBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            $this->currentJob = $job;

            try {
                if ($job->getMessage() === null) {
                    $addressEntities = $this->getDataFromMagento();

                    /** @var OrderEntity $orderEntity */
                    $orderEntity = $job->getEntity();
                    $orderEntity->setBillingAddress($addressEntities['billing'])
                        ->setShippingAddress($addressEntities['shipping']);

                    $job->setHydratedData($orderEntity->export())
                        ->setEntity($orderEntity);
                }

                if ($job->getMessage() === null) {
                    $this->jobRepository->save($job);
                }
            } catch (JobException $e) {
                $this->logger->error(
                    '[Job #' . $job->getId() . '] Error on order with Fastmag ID #'
                    . $job->getContentId() . ': ' . $e->getMessage()
                );
            } catch (CouldNotSaveException $e) {
                $this->logger->critical(
                    'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                    . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                );
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return AddressEntity[]
     *
     * @throws JobException
     */
    protected function getDataFromMagento()
    {
        $result = [];

        /** @var OrderEntity $orderEntity */
        $orderEntity = $this->currentJob->getEntity();
        $customerId = $orderEntity->getCustomerId();
        $orderId = $orderEntity->getMagentoId();

        try {
            /** @throws NoSuchEntityException|InputException */
            $order = $this->orderRepository->get($orderId);
        } catch (NoSuchEntityException $e) {
            throw new JobException(__('There\'s no order with ID #%1', $orderId));
        } catch (InputException $e) {
            throw new JobException(__('No order ID given'));
        }

        if ($order) {
            $billingAddress = $this->getAddressEntity($customerId, $order, 'billing');
            $shippingAddress = $this->getAddressEntity($customerId, $order, 'shipping');

            $result = [
                'billing'  => $billingAddress,
                'shipping' => $shippingAddress
            ];
        }

        return $result;
    }
}
