<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Order;

use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomerInterface;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Data\Rule\OrdertransactionInterface as OrdertransactionRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\CustomerFactory as CustomerEntityFactory;
use Fastmag\Sync\Process\Entity\ToFastmag\Order as OrderEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Hydration\Customer as AbstractCustomer;
use Magento\Customer\Api\CustomerRepositoryInterface as CustomerRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Newsletter\Model\Subscriber;
use Magento\Sales\Api\OrderRepositoryInterface as OrderRepository;

class Customer extends AbstractCustomer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_hydration_order_customer';

    /** @var OrderRepository $orderRepository */
    protected $orderRepository;

    /** @var Subscriber $subscriber */
    protected $subscriber;

    /** @var array $ordersToCustomers */
    protected $ordersToCustomers = [];

    /**
     * Customer constructor.
     *
     * @param Logger                   $logger
     * @param ResourceConnection       $resourceConnection
     * @param Config                   $config
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param CustomerRepository       $customerRepository
     * @param JobRepository            $jobRepository
     * @param CustomerEntityFactory    $customerEntityFactory
     * @param OrderRepository          $orderRepository
     * @param Subscriber               $subscriber
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CustomerRepository $customerRepository,
        JobRepository $jobRepository,
        CustomerEntityFactory $customerEntityFactory,
        OrderRepository $orderRepository,
        Subscriber $subscriber
    ) {
        parent::__construct(
            $logger,
            $resourceConnection,
            $config,
            $syncedCustomerRepository,
            $searchCriteriaBuilder,
            $customerRepository,
            $jobRepository,
            $customerEntityFactory
        );

        $this->orderRepository = $orderRepository;
        $this->subscriber = $subscriber;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $entities = $this->getDataFromMagento();

        if (is_array($entities)) {
            foreach ($this->ordersToCustomers as $contentId => $customerId) {
                $job = $this->getJob($contentId);

                if ($job !== null) {
                    /** @var OrderEntity $orderEntity */
                    $orderEntity = $job->getEntity();

                    $orderEntity->setCustomer($entities[$customerId]);

                    $job->setHydratedData($orderEntity->export())
                        ->setEntity($orderEntity);
                }
            }

            foreach ($this->jobs->getItems() as $job) {
                try {
                    if ($job->getMessage() === null) {
                        $this->jobRepository->save($job);
                    }
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @return CustomerEntity[]
     *
     * @throws ProcessException
     */
    protected function getDataFromMagento()
    {
        $customerIds = $this->getCustomerIds();

        foreach ($customerIds as $customerId) {
            $customerEntity = $this->customerEntityFactory->create();
            $customerEntity->setMagentoId($customerId);
            $this->entities[$customerId] = $customerEntity;
        }

        $this->getSyncedCustomers($customerIds);
        $this->getMagentoCustomersData($customerIds);

        return $this->entities;
    }

    /**
     * Get synced Fastmag ids for the current jobs
     *
     * @param int[] $customerIds
     *
     * @return void
     */
    protected function getSyncedCustomers($customerIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(SyncedCustomerInterface::MAGENTO_CUSTOMER_ID, $customerIds, 'in')
            ->create();

        $syncedCustomerList = $this->syncedCustomerRepository->getList($searchCriteria);

        foreach ($syncedCustomerList->getItems() as $syncedCustomer) {
            $customerEntity = $this->customerEntities[$syncedCustomer->getMagentoCustomerId()];
            $customerEntity->setFastmagId($syncedCustomer->getFastmagCustomerId());
        }
    }

    /**
     * Retrieve the customer IDs for the orders' jobs
     *
     * @return array
     */
    protected function getCustomerIds()
    {
        $customerIds = [];

        foreach ($this->jobs->getItems() as $job) {
            /** @var OrderEntity $orderEntity */
            $orderEntity = $job->getEntity();

            $transactionType = $orderEntity->getTransactionType();
            $orderId = $orderEntity->getMagentoId();

            if ($orderId !== null && ($transactionType === null || in_array(
                $transactionType,
                [
                    OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_SALE,
                    OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_RESERVATION,
                    OrdertransactionRule::FASTMAG_TRANSACTION_TYPE_ORDER
                ],
                true
            ))) {
                $customerId = $orderEntity->getCustomerId();
                if ($customerId === null) {
                    continue;
                }

                $this->ordersToCustomers[$job->getContentId()] = $customerId;
                $customerIds[] = $customerId;
            }
        }

        return $customerIds;
    }

    /**
     * Get Magento data for the current jobs
     *
     * @param int[] $customerIds
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getMagentoCustomersData($customerIds)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter('entity_id', $customerIds, 'in')
            ->create();

        try {
            $customersList = $this->customerRepository->getList($searchCriteria);

            foreach ($customersList->getItems() as $customer) {
                $checkSubscriber = $this->subscriber->loadByCustomerId($customer->getId());

                $customerEntity = $this->customerEntities[$customer->getId()];

                $customerEntity->setGender($customer->getGender())
                    ->setPrefix($customer->getPrefix())
                    ->setFirstname($customer->getFirstname())
                    ->setLastname($customer->getLastname())
                    ->setEmailAddress($customer->getEmail())
                    ->setCreatedAt($customer->getCreatedAt())
                    ->setStoreId($customer->getStoreId())
                    ->setDob($customer->getDob());

                $customerEntity->setSubscribeNewsletter($checkSubscriber->isSubscribed());
            }
        } catch (LocalizedException $e) {
            throw new ProcessException(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $e->getMessage(),
                implode(', ', $customerIds)
            );
        }
    }

    /**
     * Returns the job for the current order ID hydrated
     *
     * @param int $contentId
     *
     * @return Job
     */
    protected function getJob($contentId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $contentId);
    }
}
