<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-07
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration;

use Exception;
use Fastmag\Sync\Api\Data\CustomerInterfaceFactory as SyncedCustomerFactory;
use Fastmag\Sync\Api\Data\CustomerInterface as SyncedCustomer;
use Fastmag\Sync\Api\Data\Rule\StoresellerInterface as StoresellerRule;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\CustomerRepositoryInterface as SyncedCustomerRepository;
use Fastmag\Sync\Api\Rule\StoresellerRepositoryInterface as StoresellerRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Model\System\Connection\Sql\SqlInterface;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Customer
 *
 * Abstract class for Customer related Integration workers
 */
abstract class Customer extends Integration
{
    /** @var SyncedCustomerFactory $syncedCustomerFactory */
    protected $syncedCustomerFactory;

    /** @var SyncedCustomerRepository $syncedCustomerRepository */
    protected $syncedCustomerRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var StoresellerRepository $storesellerRepository */
    protected $storesellerRepository;

    /** @var Proxy $proxy */
    protected $proxy;

    /**
     * Integration constructor.
     *
     * @param Logger                   $logger
     * @param ResourceConnection       $resourceConnection
     * @param Config                   $config
     * @param JobRepository            $jobRepository
     * @param Api                      $api
     * @param SyncedCustomerFactory    $syncedCustomerFactory
     * @param SyncedCustomerRepository $syncedCustomerRepository
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     * @param StoresellerRepository    $storesellerRepository
     * @param Proxy                    $proxy
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        Api $api,
        SyncedCustomerFactory $syncedCustomerFactory,
        SyncedCustomerRepository $syncedCustomerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoresellerRepository $storesellerRepository,
        Proxy $proxy
    ) {
        parent::__construct($logger, $resourceConnection, $config, $jobRepository, $api);

        $this->syncedCustomerFactory = $syncedCustomerFactory;
        $this->syncedCustomerRepository = $syncedCustomerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storesellerRepository = $storesellerRepository;
        $this->proxy = $proxy;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $e) {
            $this->logger->notice($e->getMessage());
            return;
        }

        foreach ($this->jobs->getItems() as $job) {
            try {
                $this->processJob($job);
            } catch (JobException $e) {
                $this->logger->error(
                    '[Job #' . $job->getId() . '] Error on customer with Magento ID #' . $job->getContentId() . ': '
                    . $e->getMessage()
                );

                $job->setMessage($e->getMessage())
                    ->setTrace($e->getTraceAsString());

                $this->jobRepository->save($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ENABLE);
    }

    /**
     * Save sync customer
     *
     * @param int  $magentoId
     * @param int  $fastmagId
     * @param bool $isDeleted
     *
     * @throws JobException
     */
    protected function saveSyncCustomer($magentoId, $fastmagId, $isDeleted = false)
    {
        $searchCriteria = $this->searchCriteriaBuilder->addFilter(SyncedCustomer::MAGENTO_CUSTOMER_ID, $magentoId)
            ->create();

        $alreadySyncedCustomers = $this->syncedCustomerRepository->getList($searchCriteria);

        if ($alreadySyncedCustomers->getTotalCount() > 1) {
            throw new JobException(
                __('Customer #%1 has already been synced with Fastmag and have multiple Fastmag ID', $magentoId)
            );
        }

        if ($alreadySyncedCustomers->getTotalCount() === 1) {
            try {
                $syncCustomer = $this->syncedCustomerRepository->getByMagentoId($magentoId);
                $syncCustomer->setFastmagCustomerId($fastmagId)
                    ->setIsDeleted($isDeleted);
            } catch (Exception $e) {
                throw new JobException(
                    __('Unable to get customer #%1 Fastmag ID on synced customers table', $magentoId)
                );
            }
        } else {
            $syncCustomer = $this->syncedCustomerFactory->create();
            $syncCustomer->setMagentoCustomerId($magentoId)
                ->setFastmagCustomerId($fastmagId)
                ->setIsDeleted($isDeleted);
        }

        try {
            $this->syncedCustomerRepository->save($syncCustomer);
        } catch (CouldNotSaveException $e) {
            throw new JobException(__(
                'Unable to save customer\'s Fastmag ID: %1',
                $e->getMessage()
            ));
        }
    }

    /**
     * Check if customer email exists in Fastmag and returns Fastmag ID
     *
     * @param string $email
     *
     * @return int|null
     *
     * @throws JobException
     */
    protected function getFastmagIdByEmail($email)
    {
        $result = null;

        try {
            $sql = 'SELECT c.Client AS fastmag_id
                FROM (
                    SELECT MAX(TotalCA) AS MaxTotalCA, Email
                    FROM clients
                    WHERE Email = ' . $this->getSqlConnection()->escape($email) . '
                    GROUP BY Email
                ) AS ce INNER JOIN clients AS c
                    ON c.Email = ce.Email AND c.TotalCA = ce.MaxTotalCA';

            $row = $this->getSqlConnection()->get($sql);

            if (count($row) > 0) {
                $result = (int)$row[0]['fastmag_id'];
            }

            return $result;
        } catch (Exception $e) {
            throw new JobException(__(
                'Error when trying to get customer ID by email. Message: %1. Customer email: %2',
                $e->getMessage(),
                $email
            ));
        }
    }

    /**
     * Check if the SQL connection to Fastmag is working
     *
     * @return SqlInterface|false
     *
     * @throws NoConnectionException
     */
    protected function getSqlConnection()
    {
        $result = false;

        $connection = $this->proxy->getConnection();
        if ($connection->connect()) {
            $result = $connection;
        }

        return $result;
    }

    /**
     * Get Fastmag shop, and return the first one if there is no rule defined
     *
     * @param int $magentoStoreId
     *
     * @return string
     */
    protected function getFastmagShop($magentoStoreId)
    {
        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(StoresellerRule::STORE_ID, $magentoStoreId)
            ->create();
        $rules = $this->storesellerRepository->getList($searchCriteria);

        $result = '';

        if ($rules->getTotalCount() > 0) {
            $items = $rules->getItems();
            $result = reset($items)->getFastmagShop();
        } else {
            $searchCriteria = $this->searchCriteriaBuilder->setPageSize(1)->create();
            $rules = $this->storesellerRepository->getList($searchCriteria);

            if ($rules->getTotalCount() > 0) {
                $items = $rules->getItems();
                $result = reset($items)->getFastmagShop();
            }
        }

        return $result;
    }
}
