<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-04
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer\Address;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToFastmagRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Api;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Model\System\Connection\Sql\SqlInterface;
use Fastmag\Sync\Process\Entity\ToFastmag\Address as AddressEntity;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Address as AbstractAddress;
use Magento\Framework\App\ResourceConnection;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends AbstractAddress
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_address_save';

    /** @var Proxy $proxy */
    protected $proxy;

    /**
     * Save constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Config             $config
     * @param JobRepository      $jobRepository
     * @param Api                $api
     * @param Proxy              $proxy
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        Api $api,
        Proxy $proxy
    ) {
        parent::__construct($logger, $resourceConnection, $config, $jobRepository, $api);

        $this->proxy = $proxy;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $e) {
            $this->logger->notice($e->getMessage());
            return;
        }

        foreach ($this->jobs->getItems() as $job) {
            try {
                $this->processJob($job);
            } catch (JobException $e) {
                $this->logger->error(
                    '[Job #' . $job->getId() . '] Error on customer\'s address with Magento ID #' . $job->getContentId()
                    . ': ' . $e->getMessage()
                );

                $job->setMessage($e->getMessage())
                    ->setTrace($e->getTraceAsString());

                $this->jobRepository->save($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ENABLE);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $customerEntity */
        $customerEntity = $job->getEntity();

        if ($customerEntity->getShippingAddress() !== null) {
            $shippingAddress = $customerEntity->getShippingAddress();

            $this->sendFastmagAddress(
                $customerEntity->getMagentoId(),
                $customerEntity->getFastmagId(),
                $shippingAddress
            );
        }
    }

    /**
     * Send address to Fastmag
     *
     * @param int           $magentoId
     * @param int           $fastmagId
     * @param AddressEntity $addressEntity
     *
     * @throws JobException
     */
    protected function sendFastmagAddress($magentoId, $fastmagId, $addressEntity)
    {
        $addressExists = $this->checkFastmagAddress($fastmagId, $addressEntity->getAlias());

        if (!$addressExists) {
            $this->createFastmagAddress($magentoId, $fastmagId, $addressEntity);
        } else {
            $this->updateFastmagAddress($magentoId, $fastmagId, $addressEntity);
        }
    }

    /**
     * Check if address exists on Fastmag
     *
     * @param int    $fastmagId
     * @param string $alias
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function checkFastmagAddress($fastmagId, $alias)
    {
        try {
            $sql = 'SELECT AdrLivraison
                FROM clientadresse
		        WHERE AdrLivraison = ' . $this->getSqlConnection()->escape(utf8_decode($alias));

            $rows = $this->getSqlConnection()->get($sql);

            return (bool)count($rows);
        } catch (Exception $e) {
            throw new JobException(__(
                'Error when trying to check customer address on Fastmag. ' .
                'Message: %1. Customer ID: %2. Address alias: %3',
                $e->getMessage(),
                $fastmagId,
                $alias
            ));
        }
    }

    /**
     * Check if the SQL connection to Fastmag is working
     *
     * @return SqlInterface|false
     *
     * @throws NoConnectionException
     */
    protected function getSqlConnection()
    {
        $result = false;

        $connection = $this->proxy->getConnection();
        if ($connection->connect()) {
            $result = $connection;
        }

        return $result;
    }

    /**
     * Create address on Fastmag
     *
     * @param int           $magentoId
     * @param int           $fastmagId
     * @param AddressEntity $addressEntity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function createFastmagAddress($magentoId, $fastmagId, $addressEntity)
    {
        $ediData = [
            $fastmagId,
            utf8_decode(Text::upper($addressEntity->getAlias())),
            utf8_decode($addressEntity->getLastname() . ' ' . $addressEntity->getFirstname()),
            utf8_decode($addressEntity->getStreetOne()),
            utf8_decode($addressEntity->getStreetTwo()),
            utf8_decode($addressEntity->getPostcode()),
            utf8_decode($addressEntity->getCity()),
            utf8_decode($addressEntity->getCountryId()),
            '', // Bâtiment
            '', // Digicode 1
            '', // Digicode 2
            '', // Interphone
            '', // Etage
            '', // Porte
            '', // Escalier
            '', // Ascenseur
            utf8_decode($addressEntity->getCompany()),
            utf8_decode($addressEntity->getPhoneNumber()),
            '', // Observation
            $addressEntity->isArchived()
        ];

        try {
            $ediFormattedData = $this->edi->formatInline(Edi::EDI_LINE_ADDRESS_ID, $ediData);

            $this->edi->post($ediFormattedData);
        } catch (NoConnectionException $e) {
            throw new JobException(__(
                'Unable to send address\' data of the customer #%1 to Fastmag through EDI: %2. Data sent: %3',
                $magentoId,
                $e->getMessage(),
                $ediData
            ));
        }
    }

    /**
     * Update address on Fastmag
     *
     * @param int           $magentoId
     * @param int           $fastmagId
     * @param AddressEntity $addressEntity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagAddress($magentoId, $fastmagId, $addressEntity)
    {
        $this->createFastmagAddress($magentoId, $fastmagId, $addressEntity);
    }
}
