<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Order
 *
 * Hydration class used for inserting and updating orders from Fastmag to Magento
 */
class Order extends Hydration
{
    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * Hydration constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param Config             $config
     * @param Proxy              $proxy
     * @param JobRepository      $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        Proxy $proxy,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $config, $proxy);

        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $results = $this->getDataFromFastmag();

        if (is_array($results)) {
            foreach ($results as $row) {
                $job = $this->getJob($row['order_id']);

                if ($job !== null) {
                    $job->setHydratedData($row);
                }
            }

            foreach ($this->jobs->getItems() as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $e) {
                    $this->logger->critical(
                        'Can not save job ' . $job->getJobCode() . ' for the entity #' . $job->getContentId()
                        . ' (job #' . $job->getId() . '): ' . $e->getMessage() . "\n" . $e->getTraceAsString()
                    );
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $ordersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT Vente AS order_id, Client AS customer_id, Date AS creation_date, Heure AS creation_time,
                Total AS total, CodeMag AS shop_code, VShopStatus AS status, ListeColis AS tracking_number,
                (
                    SELECT TRIM(valeurstring)
                    FROM complementsr AS csr
                    WHERE csr.Champ = \'NOM_WEB\' AND csr.Nature = \'MAGASINS\' AND csr.Clef = v.CodeMag
                ) AS alt_shop_code,
                (SELECT r.Libelle FROM reglements AS r WHERE r.Vente = v.Vente LIMIT 1) AS payment_method
                FROM ventes AS v
                WHERE v.Nature = \'VENTE\'
                    AND v.Client > 0
                    AND v.CodeMag NOT IN (\'CENTRAL\')
                    AND (v.InfosComp IS NULL OR v.InfosComp NOT LIKE \'Magento%\')
                    AND v.Vente IN (' . $this->getConnection()->escape($ordersIds) . ')';

            return $this->getConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException(
                'Error when hydrating orders. Message: %1. Orders IDs: %2',
                $e->getMessage(),
                implode(', ', $ordersIds)
            );
        }
    }

    /**
     * Returns the job for the current order ID hydrated
     *
     * @param int $orderFastmagId
     *
     * @return Job
     */
    protected function getJob($orderFastmagId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $orderFastmagId);
    }
}
