<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-04-06
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product\Reset;

use Exception;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\StoresellerRepositoryInterface as StoresellerRuleRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Price
 *
 * Hydration class used for reseting products prices from Fastmag to Magento
 */
class Price extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_reset_price';

    /** @var StoresellerRuleRepository $storesellerRuleRepository */
    protected $storesellerRuleRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * Product constructor
     *
     * @param Logger                    $logger
     * @param ResourceConnection        $resourceConnection
     * @param Config                    $config
     * @param Proxy                     $proxy
     * @param StoresellerRuleRepository $storesellerRepository
     * @param SearchCriteriaBuilder     $searchCriteriaBuilder
     * @param JobRepository             $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        Proxy $proxy,
        StoresellerRuleRepository $storesellerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $config, $proxy);

        $this->storesellerRuleRepository = $storesellerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            try {
                $hydratedData = $this->getDataFromFastmag();
                $job->setHydratedData($hydratedData);
            } catch (JobException $e) {
                $job->setMessage($e->getMessage())
                    ->setTrace($e->getTraceAsString());
            }

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $e) {
                $this->logger->critical(
                    'Can not save job ' . $job->getJobCode() . ' (job #' . $job->getId() . '): '
                    . $e->getMessage() . "\n" . $e->getTraceAsString()
                );
            }
        }
    }

    /**
     * Get all catalog's products prices
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getDataFromFastmag()
    {
        $result = [];

        $searchCriteria = $this->searchCriteriaBuilder->create();
        $storesellerRules = $this->storesellerRuleRepository->getList($searchCriteria);

        foreach ($storesellerRules->getItems() as $rule) {
            $fastmagShop = $rule->getFastmagShop();
            $rateCode = $rule->getRateCode();

            if (!array_key_exists($fastmagShop, $result)) {
                $result[$fastmagShop] = [];
            }

            if (!array_key_exists($rateCode, $result[$fastmagShop])) {
                $sqlSelect = '
                SELECT s.BarCode  AS sku,
                   s.Couleur      AS color,
                   s.Taille       AS size,
                   p.Produit      AS fastmag_product_id,
                   pf.PrixVente   AS indicative_price,
                   s.PrixVente    AS standard_price,
                   p.PrixVente    AS combination_price,
                   pr.Prix        AS discount_special_price,
                   pr.Remise      AS discount_percent,
                   pr.DateDebut   AS discount_begin_at,
                   pr.DateFin     AS discount_end_at,
                   pr.Motif       AS discount_reason,
                   prtc.Prix      AS combination_discount_special_price,
                   prtc.Remise    AS combination_discount_percent,
                   prtc.DateDebut AS combination_discount_begin_at,
                   prtc.DateFin   AS combination_discount_end_at,
                   prtc.Motif     AS combination_discount_reason';

                if ($rateCode !== null) {
                    $sqlSelect .= ',
                    pt.Prix        AS rate_price';
                }

                $sqlFrom = '
                FROM produitsfiches AS pf
                     LEFT JOIN stock AS s ON pf.BarCode = s.BarCode
                     LEFT JOIN produits AS p ON s.BarCode = p.BarCode AND s.Couleur = p.Couleur AND s.Taille = p.Taille
                     LEFT JOIN prixremise AS pr
                               ON s.BarCode = pr.BarCode AND s.CodeMag = pr.CodeMag AND pr.DateDebut <= date(NOW())
                                   AND pr.DateFin >= date(NOW())
                     LEFT JOIN prixremisetc AS prtc
                               ON s.BarCode = prtc.BarCode AND s.Couleur = prtc.Couleur AND s.Taille = prtc.Taille
                                   AND s.CodeMag = prtc.CodeMag AND prtc.DateDebut <= date(NOW())
                                   AND prtc.DateFin >= date(NOW()) AND (prtc.Remise > 0 OR prtc.Prix > 0)';

                if ($rateCode !== null) {
                    $sqlFrom .= '
                     LEFT JOIN produitstarifs AS pt
                               ON s.BarCode = pt.BarCode AND s.Couleur = pt.Couleur AND s.Taille = pt.Taille
                                      AND pt.Tarif = "' . $rateCode . '"';
                }

                $sqlWhere = '
                WHERE s.AR = 1
                  AND s.CodeMag = "' . $fastmagShop . '"';

                if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB)) {
                    $sqlWhere .= '
                  AND pf.Actif = 1 AND pf.VisibleWeb = 1';
                }

                $sqlGroupBy = '
                GROUP BY s.Barcode, s.Couleur, s.Taille';

                try {
                    $rows = $this->getConnection()->get($sqlSelect . $sqlFrom . $sqlWhere . $sqlGroupBy);
                } catch (Exception $e) {
                    throw new JobException(__('Unable to get all prices data from Fastmag: %1', $e->getMessage()));
                }

                $ruleResults = $this->processRows($rows);

                $result[$fastmagShop][$rateCode] = $ruleResults;
            }
        }

        return $result;
    }

    /**
     * Process rows to reset. The point here is minimize the size taken by the hydrated data
     *
     * @param array $rows
     *
     * @return array
     */
    protected function processRows($rows)
    {
        $result = [];

        $currentSku = '';
        $currentSkuPrices = [];
        $currentSkuSpecialPrices = [];

        foreach ($rows as $row) {
            if ($currentSku !== $row['sku']) {
                if (count($currentSkuPrices) !== 0) {
                    $result[$currentSku] = [
                        'price' => $currentSkuPrices
                    ];

                    if ($currentSkuSpecialPrices !== []) {
                        $result[$currentSku]['special_price'] = $currentSkuSpecialPrices;
                    }
                }

                $currentSku = $row['sku'];
                $currentSkuPrices = [];
                $currentSkuSpecialPrices = [];
            }

            $childPrice = $this->getChildPrice(
                $row['indicative_price'],
                $row['standard_price'],
                $row['combination_price'],
                ($row['rate_price'] ?? null)
            );

            $specialPriceData = [
                'special_price' => $row['discount_special_price'],
                'percent'       => $row['discount_percent'],
                'begin_at'      => $row['discount_begin_at'],
                'end_at'        => $row['discount_end_at'],
                'reason'        => $row['discount_reason']
            ];

            if ($currentSkuPrices === []) {
                $currentSkuPrices['default'] = $childPrice;
            }

            if ($currentSkuPrices['default'] !== $childPrice) {
                $currentSkuPrices[$row['fastmag_product_id']] = $childPrice;
            }

            $specialPriceTcData = [
                'special_price' => $row['combination_discount_special_price'],
                'percent'       => $row['combination_discount_percent'],
                'begin_at'      => $row['combination_discount_begin_at'],
                'end_at'        => $row['combination_discount_end_at'],
                'reason'        => $row['combination_discount_reason']
            ];

            $childSpecialPricesData = $this->getChildSpecialPriceData($specialPriceData, $specialPriceTcData);

            if ($childSpecialPricesData !== null) {
                if ($currentSkuSpecialPrices === []) {
                    $currentSkuSpecialPrices['default'] = $childSpecialPricesData;
                }

                if ($currentSkuSpecialPrices['default'] !== $childSpecialPricesData) {
                    $currentSkuSpecialPrices[$row['fastmag_product_id']] = $childSpecialPricesData;
                }
            }
        }

        return $result;
    }

    /**
     * Get correct child price
     *
     * @param float      $indicativePrice
     * @param float      $combinationPrice
     * @param float      $standardPrice
     * @param float|null $ratePrice
     *
     * @return float
     */
    public function getChildPrice($indicativePrice, $combinationPrice, $standardPrice, $ratePrice = null)
    {
        $result = $indicativePrice;

        if ($ratePrice !== null) {
            $result = $ratePrice;
        } elseif ($combinationPrice > 0) {
            $result = $combinationPrice;
        } elseif ($standardPrice > 0) {
            $result = $standardPrice;
        }

        return $result;
    }

    /**
     * Get correct child special price
     *
     * @param array $specialPriceData
     * @param array $specialPriceTcData
     *
     * @return array
     */
    public function getChildSpecialPriceData($specialPriceData, $specialPriceTcData)
    {
        $result = null;

        if ($specialPriceTcData['special_price'] !== null) {
            $result = $specialPriceTcData;
        } elseif ($specialPriceData['special_price'] !== null) {
            $result = $specialPriceData;
        }

        return $result;
    }
}
