<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-05-17
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Customer\Address;

use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\Constants;
use Fastmag\Sync\Model\Jobqueue\ToMagento as Job;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository as JobRepository;
use Fastmag\Sync\Process\Entity\ToMagento\Customer as CustomerEntity;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\Address as AddressEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Address\Save as DefaultSave;
use Magento\Customer\Api\Data\AddressInterface;
use Magento\Customer\Api\AddressRepositoryInterface;
use Magento\Customer\Api\Data\AddressInterfaceFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\LocalizedException;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_customer_create and tomagento_integration_customer_update jobs
 * Deals with customer address creation and update
 */
class Save extends DefaultSave
{
    /** @inheritDoc */
    protected $code = 'tomagento_integration_customer_address_save';

    /** @var AddressRepositoryInterface $addressRepository */
    protected $addressRepository;

    /** @var AddressInterfaceFactory $addressFactory */
    protected $addressFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /**
     * Save constructor
     *
     * @param Logger                     $logger
     * @param ResourceConnection         $resourceConnection
     * @param Config                     $config
     * @param JobRepository              $jobRepository
     * @param AddressRepositoryInterface $addressRepository
     * @param AddressInterfaceFactory    $addressFactory
     * @param SearchCriteriaBuilder      $searchCriteriaBuilder
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        Config $config,
        JobRepository $jobRepository,
        AddressRepositoryInterface $addressRepository,
        AddressInterfaceFactory $addressFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct($logger, $resourceConnection, $config, $jobRepository);

        $this->addressRepository = $addressRepository;
        $this->addressFactory = $addressFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $e) {
            $this->logger->notice($e->getMessage());
            return;
        }

        foreach ($this->jobs->getItems() as $job) {
            try {
                $this->processJob($job);
            } catch (JobException $e) {
                $this->logger->error(
                    '[Job #' . $job->getId() . '] Error on customer with Fastmag ID #' . $this->getJobEntityId($job)
                    . ': ' . $e->getMessage()
                );

                $job->setMessage($e->getMessage())
                    ->setTrace($e->getTraceAsString());

                $this->jobRepository->save($job);
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function isEnabled()
    {
        return $this->config->isSetFlag(Config::XML_PATH_CUSTOMER_IMPORT_ENABLE);
    }

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $customerEntity */
        $customerEntity = $job->getEntity();

        $this->saveBillingAddress($customerEntity);

        foreach ($customerEntity->getShippingAddresses() as $shippingAddress) {
            $this->saveShippingAddress($customerEntity, $shippingAddress);
        }
    }

    /**
     * Save billing address of the customer
     *
     * @param CustomerEntity $customerEntity
     *
     * @throws JobException
     */
    protected function saveBillingAddress($customerEntity)
    {
        $addressEntity = $customerEntity->getBillingAddress();

        if ($this->isAddressValid($addressEntity)) {
            $address = $this->checkIfAddressExists($customerEntity->getMagentoId(), $addressEntity->getAlias());
            if (!$address) {
                $address = $this->addressFactory->create();
                $address->setCustomerId($customerEntity->getMagentoId());
            }

            if (!$customerEntity->getMagentoDefaultBilling()) {
                $address->setIsDefaultBilling(true);
            }

            $this->saveEntity($address, $addressEntity);
        }
    }

    /**
     * Save shipping address of the customer
     *
     * @param CustomerEntity $customerEntity
     * @param AddressEntity  $addressEntity
     *
     * @throws JobException
     */
    protected function saveShippingAddress($customerEntity, $addressEntity)
    {
        if ($this->isAddressValid($addressEntity)) {
            $address = $this->checkIfAddressExists($customerEntity->getMagentoId(), $addressEntity->getAlias());
            if (!$address) {
                $address = $this->addressFactory->create();
                $address->setCustomerId($customerEntity->getMagentoId());
            }

            if (!$customerEntity->getMagentoDefaultShipping() && $addressEntity->isLastShipping()) {
                $address->setIsDefaultShipping(true);
            }

            $this->saveEntity($address, $addressEntity);
        }
    }

    /**
     * Save address with entity attributes
     *
     * @see Presta : \JobWorker_Presta_CustomerCreate::exec
     * @see M1     : \Fastmag_Sync_Model_Worker_Magento::createOrUpdateCustomer
     *
     * @todo : find a way to use deeper parts of the core code to decrease memory usage and processing time
     *
     * @param AddressInterface $address
     * @param AddressEntity    $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function saveEntity($address, $entity)
    {
        $this->setAddressData($address, $entity);

        try {
            $this->addressRepository->save($address);
        } catch (LocalizedException $e) {
            throw new JobException(__('Unable to sync the address: %1', $e->getMessage()));
        }
    }

    /**
     * Check if address has street, postcode, city and alias filled at least
     *
     * @param AddressEntity $address
     *
     * @return bool
     *
     * @throws JobException
     */
    protected function isAddressValid($address)
    {
        $result = parent::isAddressValid($address);

        if ($address->getAlias() === null) {
            throw new JobException(__('The address given does not have alias filled'));
        }

        return $result;
    }

    /**
     * Get job's entity ID
     *
     * @param Job $job
     *
     * @return int
     */
    protected function getJobEntityId($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        return $entity->getFastmagId();
    }

    /**
     * Check if an address in Magento DB has the alias for the customer ID given in param
     * and returns it if it exists
     *
     * @param int    $customerId
     * @param string $addressAlias
     *
     * @return AddressInterface|false
     *
     * @throws JobException
     */
    protected function checkIfAddressExists($customerId, $addressAlias)
    {
        try {
            $this->searchCriteriaBuilder->addFilter('parent_id', $customerId)
                ->addFilter(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE, $addressAlias);
            $searchCriteria = $this->searchCriteriaBuilder->create();

            $addresses = $this->addressRepository->getList($searchCriteria);
        } catch (LocalizedException $e) {
            return false;
        }

        if ($addresses->getTotalCount() > 1) {
            throw new JobException(__(
                'It seems there is more than one address matching the alias "%1". Please review them ASAP',
                $addressAlias
            ));
        }

        if ($addresses->getTotalCount() === 0) {
            $address = false;
        } else {
            $address = $addresses->getItems()[0];
        }

        return $address;
    }

    /**
     * Set entity data into address model
     *
     * @param AddressInterface $address
     * @param AddressEntity    $entity
     *
     * @return void
     */
    protected function setAddressData($address, $entity)
    {
        $address->setPrefix($entity->getPrefix())
            ->setFirstname($entity->getFirstname())
            ->setLastname($entity->getLastname())
            ->setCompany($entity->getCompany())
            ->setStreet($entity->getStreet())
            ->setPostcode($entity->getPostcode())
            ->setCity($entity->getCity())
            ->setCountryId($entity->getContryId())
            ->setTelephone($entity->getPhoneNumber());

        $address->setCustomAttribute(Constants::ATTRIBUTE_ADDRESS_ALIAS_CODE, $entity->getAlias());
    }
}
