<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-05-31
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection\Sql;

use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config\Provider\DirectSql as DirectSqlConfig;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\DB\Adapter\Pdo\Mysql as MysqlAdapter;
use Magento\Framework\DB\Adapter\Pdo\MysqlFactory as MysqlAdapterFactory;
use Magento\Framework\Model\ResourceModel\Type\Db\Pdo\Mysql as MysqlResource;
use PDO;

/**
 * Class DirectSql
 *
 * Class to connect and query to Fastmag through direct SQL interfaces
 */
class DirectSql extends MysqlResource implements SqlInterface
{
    /* @var MysqlAdapter */
    protected $connection;

    /** @var DirectSqlConfig */
    protected $directSqlConfigProvider;

    /* @var MysqlAdapterFactory */
    protected $connectionFactory;

    /** @var Logger */
    protected $logger;

    /** @var ScopeConfigInterface */
    protected $scopeConfig;

    /**
     * DirectSql constructor.
     *
     * @param DirectSqlConfig          $directSqlConfigProvider
     * @param Logger                   $logger
     * @param ScopeConfigInterface     $scopeConfig
     * @param MysqlAdapterFactory|null $mysqlFactory
     */
    public function __construct(
        DirectSqlConfig $directSqlConfigProvider,
        Logger $logger,
        ScopeConfigInterface $scopeConfig,
        MysqlAdapterFactory $mysqlFactory = null
    ) {
        $this->directSqlConfigProvider = $directSqlConfigProvider;
        $this->logger = $logger;
        $this->scopeConfig = $scopeConfig;

        $config = $this->getConfigFromProvider();

        parent::__construct($config, $mysqlFactory);
    }

    /**
     * @inheritDoc
     */
    public function connect()
    {
        if ($this->connection === null) {
            $this->connection = $this->getConnection();
        }

        return $this->connection !== null;
    }

    /**
     * @inheritDoc
     */
    public function get($sql)
    {
        try {
            $result = $this->connection->query($sql);
        } catch (Exception $e) {
            throw new NoConnectionException($e);
        }

        return $result === null ? [] : $result->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * @inheritDoc
     */
    public function post($sql)
    {
        $result = true;

        try {
            $result = $this->connection->query($sql) !== null;
        } catch (Exception $e) {
            throw new NoConnectionException($e);
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function multiPost($sql)
    {
        $result = true;

        try {
            $result = $this->connection->multiQuery($sql) !== null;
        } catch (Exception $e) {
            throw new NoConnectionException($e);
        }

        return $result;
    }

    /**
     * Get Direct SQL config stored in conf
     *
     * @return array
     */
    protected function getConfigFromProvider()
    {
        return [
            'host'     => $this->directSqlConfigProvider->getHost() . ':' . $this->directSqlConfigProvider->getPort(),
            'username' => $this->directSqlConfigProvider->getUser(),
            'password' => $this->directSqlConfigProvider->getPassword(),
            'dbname'   => $this->directSqlConfigProvider->getDatabase()
        ];
    }

    /**
     * @inheritDoc
     */
    public function escape($value, $forceString = false)
    {
        if (is_array($value)) {
            $result = '';

            foreach ($value as $row) {
                $result .= $this->escape($row, $forceString) . ', ';
            }

            $result = trim($result, ', ');
        } else {
            $result = stripslashes($value);

            if ($forceString || !is_numeric($result)) {
                $search = ['\\', "\0", "\n", "\r", "\x1a", "'", '"'];
                $replace = ['\\\\', '\\0', '\\n', '\\r', "\Z", "\'", '\"'];

                $result = '\'' . strip_tags(nl2br(str_replace($search, $replace, $result))) . '\'';
            }
        }

        return $result;
    }
}
