<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToMagento;

use DateTime;
use Exception;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Helper\Text;
use Fastmag\Sync\Model\Config\Source\Customer\GroupField;
use Fastmag\Sync\Process\Entity\ToMagento\Customer\Address;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Gender;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Prefix;
use Magento\Customer\Api\Data\GroupInterface;
use Magento\Customer\Api\Data\GroupInterfaceFactory;
use Magento\Customer\Api\GroupRepositoryInterface;
use Magento\Customer\Model\ResourceModel\GroupRepository;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\DB\Adapter\Pdo\Mysql;
use Magento\Framework\Exception\LocalizedException;
use Zend_Validate;
use Zend_Validate_Exception;

/**
 * Class Customer
 *
 * Entity class for customer.
 * Save and transform data from Fastmag for Magento easy integration,
 * and transfer it from hydration worker to integration worker
 *
 * @deprecated Data formating will be done on integration worker
 */
class Customer
{
    /** @var string */
    public const XML_CONFIG_HYDRATION_SIZE = 'fastmag_sync_customers/import/customer_groups_fields';

    /** @var string */
    public const CUSTOMER_DISCOUNT_GROUP_PREFIX = 'fm_discount_';

    /** @var string */
    public const CUSTOMER_REGROUPING_GROUP_PREFIX = 'fm_regrouping_';

    /** @var string $fastmagId */
    protected $fastmagId;

    /** @var string $magentoId */
    protected $magentoId;

    /** @var string $isActive */
    protected $isActive;

    /** @var string $gender */
    protected $gender;

    /** @var string $prefix */
    protected $prefix;

    /** @var string $firstname */
    protected $firstname;

    /** @var string $lastname */
    protected $lastname;

    /** @var string $emailAddress */
    protected $emailAddress;

    /** @var string $birthdayYear */
    protected $birthdayYear;

    /** @var string $birthdayMonth */
    protected $birthdayMonth;

    /** @var string $birthdayDay */
    protected $birthdayDay;

    /** @var string $discountRate */
    protected $discountRate;

    /** @var string $subscribeNewsletter */
    protected $subscribeNewsletter;

    /** @var string $regrouping */
    protected $regrouping;

    /** @var string $creationDate */
    protected $creationDate;

    /** @var Address $billingAddress */
    protected $billingAddress;

    /** @var Address[] $shippingAddresses */
    protected $shippingAddresses;

    /** @var int $magentoDefaultBilling */
    protected $magentoDefaultBilling;

    /** @var int $magentoDefaultShipping */
    protected $magentoDefaultShipping;

    /** @var ScopeConfigInterface $scopeConfig */
    protected $scopeConfig;

    /** @var GroupRepositoryInterface $groupRepository */
    protected $groupRepository;

    /** @var GroupInterfaceFactory $customerGroupeFactory */
    protected $customerGroupFactory;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /**
     * Customer constructor.
     *
     * @param ScopeConfigInterface     $scopeConfig
     * @param GroupRepositoryInterface $customerGroupRepository
     * @param GroupInterfaceFactory    $customerGroupFactory
     * @param SearchCriteriaBuilder    $searchCriteriaBuilder
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        GroupRepositoryInterface $customerGroupRepository,
        GroupInterfaceFactory $customerGroupFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->groupRepository = $customerGroupRepository;
        $this->customerGroupFactory = $customerGroupFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @inheritDoc
     */
    public function getFastmagId()
    {
        return (int)$this->fastmagId;
    }

    /**
     * @inheritDoc
     */
    public function getMagentoId()
    {
        return (int)$this->magentoId;
    }

    /**
     * @inheritDoc
     */
    public function getIsActive()
    {
        return (int)$this->isActive === 1;
    }

    /**
     * @inheritDoc
     */
    public function getGender()
    {
        return Gender::getMagentoValue(Text::upper($this->gender));
    }

    /**
     * @inheritDoc
     */
    public function getPrefix()
    {
        return Prefix::getMagentoValue(Text::upper($this->prefix));
    }

    /**
     * @inheritDoc
     */
    public function getFirstname()
    {
        $result = $this->firstname;

        if (empty($this->firstname) && !empty($this->lastname)) {
            $names = explode(' ', $this->lastname);
            $result = array_pop($names);
        }

        return Text::capitalize($result);
    }

    /**
     * @inheritDoc
     */
    public function getLastname()
    {
        $result = $this->lastname;

        if (empty($this->firstname) && !empty($this->lastname)) {
            $names = explode(' ', $this->lastname);
            array_pop($names);
            $result = implode(' ', $names);

        }

        return Text::capitalize($result);
    }

    /**
     * @inheritDoc
     */
    public function getEmailAddress()
    {
        try {
            $result = false;

            if (Zend_Validate::is($this->emailAddress, 'EmailAddress')) {
                $result = Text::lower($this->emailAddress);
            }
        } catch (Zend_Validate_Exception $e) {
            $result = false;
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function getBirthday()
    {
        try {
            $date = new DateTime($this->birthdayYear . '-' . $this->birthdayMonth . '-' . $this->birthdayDay);
            $result = $date->format(Mysql::DATE_FORMAT);
        } catch (Exception $e) {
            $result = null;
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function getDiscountRate()
    {
        return (int)$this->discountRate;
    }

    /**
     * @inheritDoc
     */
    public function getSubscribeNewsletter()
    {
        return (bool)$this->subscribeNewsletter;
    }

    /**
     * @inheritDoc
     */
    public function getCustomerGroupName()
    {
        $config = $this->scopeConfig->getValue(self::XML_CONFIG_HYDRATION_SIZE);
        $customerGroup = 'General';

        if (!empty($this->discountRate)
            && in_array($config, [GroupField::VALUE_DISCOUNT, GroupField::VALUE_BOTH], true)
        ) {
            $customerGroup = self::CUSTOMER_DISCOUNT_GROUP_PREFIX . $this->discountRate;
        }

        if (!empty($this->regrouping)
            && in_array($config, [GroupField::VALUE_REGROUPING, GroupField::VALUE_BOTH], true)
        ) {
            $customerGroup = self::CUSTOMER_REGROUPING_GROUP_PREFIX . $this->regrouping;
        }

        return $customerGroup;
    }

    /**
     * @inheritDoc
     */
    public function getCustomerGroupId()
    {
        $name = $this->getCustomerGroupName();

        $this->searchCriteriaBuilder->addFilter(GroupInterface::CODE, $name);
        $searchCriteria = $this->searchCriteriaBuilder->create();

        try {
            $customerGroupList = $this->groupRepository->getList($searchCriteria);
        } catch (LocalizedException $e) {
            throw new JobException(__(
                'Unable to search for the customer group "%1": %2',
                $name,
                $e->getMessage()
            ));
        }

        if ($customerGroupList->getTotalCount() > 1) {
            throw new JobException(__(
                'It seems there is more than one customer group called "%1". Please review them ASAP.',
                $name
            ));
        }

        if ($customerGroupList->getTotalCount() === 0) {
            $customerGroup = $this->customerGroupFactory->create();
            $customerGroup->setCode($name)
                ->setTaxClassId(GroupRepository::DEFAULT_TAX_CLASS_ID);

            try {
                $this->groupRepository->save($customerGroup);
            } catch (LocalizedException $e) {
                throw new JobException(__(
                    'Unable to save the new customer group "%1": %2',
                    $name,
                    $e->getMessage()
                ));
            }

            $result = $customerGroup->getId();
        } else {
            $result = $customerGroupList->getItems()[0]->getId();
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function getCreatedAt()
    {
        try {
            $date = new DateTime($this->creationDate);
            $result = $date->format(Mysql::DATETIME_FORMAT);
        } catch (Exception $e) {
            $result = date(Mysql::DATETIME_FORMAT);
        }

        return $result;
    }

    /**
     * @inheritDoc
     *
     * @return Address
     */
    public function getBillingAddress()
    {
        return $this->billingAddress;
    }

    /**
     * @inheritDoc
     *
     * @return Address[]
     */
    public function getShippingAddresses()
    {
        return $this->shippingAddresses;
    }

    /**
     * @inheritDoc
     */
    public function getMagentoDefaultBilling()
    {
        return $this->magentoDefaultBilling;
    }

    /**
     * @inheritDoc
     */
    public function getMagentoDefaultShipping()
    {
        return $this->magentoDefaultShipping;
    }

    /**
     * @inheritDoc
     */
    public function setFastmagId($fastmagId)
    {
        $this->fastmagId = $fastmagId;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setMagentoId($magentoId)
    {
        $this->magentoId = $magentoId;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setIsActive($isActie)
    {
        $this->isActive = $isActie;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setGender($gender)
    {
        $this->gender = $gender;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setPrefix($prefix)
    {
        $this->prefix = $prefix;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setFirstname($firstname)
    {
        $this->firstname = $firstname;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setLastname($lastname)
    {
        $this->lastname = $lastname;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setEmailAddress($emailAddress)
    {
        $this->emailAddress = $emailAddress;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setBirthdayYear($year)
    {
        $this->birthdayYear = $year;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setBirthdayMonth($month)
    {
        $this->birthdayMonth = $month;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setBirthdayDay($day)
    {
        $this->birthdayDay = $day;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setDiscountRate($discountRate)
    {
        $this->discountRate = $discountRate;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setSubscribeNewsletter($subscribeNewsletter)
    {
        $this->subscribeNewsletter = $subscribeNewsletter;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setRegrouping($regrouping)
    {
        $this->regrouping = $regrouping;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setCreationDate($creationDate)
    {
        $this->creationDate = $creationDate;

        return $this;
    }

    /**
     * @inheritDoc
     *
     * @param Address $address
     */
    public function setBillingAddress($address)
    {
        $this->billingAddress = $address;

        return $this;
    }

    /**
     * @inheritDoc
     *
     * @param Address[] $address
     */
    public function setShippingAddresses($addresses)
    {
        $this->shippingAddresses = $addresses;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setMagentoDefaultBilling($addressId)
    {
        $this->magentoDefaultBilling = $addressId;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function setMagentoDefaultShipping($addressId)
    {
        $this->magentoDefaultShipping = $addressId;

        return $this;
    }
}
