<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-21
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToFastmagInterface as Job;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Entity\ToFastmag\Customer as CustomerEntity;
use Fastmag\Sync\Process\Worker\ToFastmag\Integration\Customer;
use stdClass;

/**
 * Class Save
 *
 * Integration class used for inserting or updating customers from Magento to Fastmag
 */
class Save extends Customer
{
    /** @inheritDoc */
    protected $code = 'tofastmag_integration_customer_save';

    /** @inheritDoc */
    protected $subordinateWorkersCodes = ['tofastmag_integration_customer_address_save'];

    /**
     * Process job
     *
     * @param Job $job
     *
     * @return void
     *
     * @throws JobException
     */
    protected function processJob($job)
    {
        /** @var CustomerEntity $entity */
        $entity = $job->getEntity();

        if ($entity->getFastmagId() === null) {
            $fastmagId = $this->getFastmagIdByEmail($entity->getEmailAddress());

            if ($fastmagId) {
                $entity->setFastmagId($fastmagId);

                $this->saveSyncCustomer($entity->getMagentoId(), $fastmagId);
            }
        }

        $newFastmagId = $this->sendDataToFastmag($entity->getMagentoId(), $entity);
        if ($newFastmagId !== false) {
            $entity->setFastmagId($newFastmagId);

            $this->saveSyncCustomer($entity->getMagentoId(), $newFastmagId);
        }
    }

    /**
     * Send customer to Fastmag
     *
     * @param int            $customerId
     * @param CustomerEntity $customerEntity
     *
     * @return int|bool new Fastmag ID if it's an insertion, false otherwise
     *
     * @throws JobException
     */
    protected function sendDataToFastmag($customerId, $customerEntity)
    {
        $result = false;

        if ($customerEntity->getFastmagId() === null) {
            $result = $this->createFastmagCustomer($customerId, $customerEntity);
        } else {
            $this->updateFastmagCustomer($customerId, $customerEntity);
        }

        return $result;
    }

    /**
     * Create customer on Fastmag
     *
     * @param int            $magentoId
     * @param CustomerEntity $entity
     *
     * @return int
     *
     * @throws JobException
     */
    protected function createFastmagCustomer($magentoId, $entity)
    {
        $billingAddress = $entity->getBillingAddress();

        if ($billingAddress === null) {
            throw new JobException(__(
                'Unable to create customer #%1 on Fastmag: no billing address found.',
                $entity->getMagentoId()
            ));
        }

        $customerData = $this->setDataRequest($entity);

        try {
            $response = $this->api->post('/boa/client/create/index.ips', $customerData);

            if (!array_key_exists('data', $response)) {
                throw new ApiException($this->api->getLastRequest(), 'No data received from Fastmag');
            }
            $response = $response['data'][0];

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException($this->api->getLastRequest(), $message);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to create customer #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $magentoId,
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }

        return (int)$response['Client'];
    }

    /**
     * Set the customer entity as intented for the API
     *
     * @param CustomerEntity $entity
     * @param bool           $update
     *
     * @return stdClass
     */
    protected function setDataRequest($entity, $update = false)
    {
        $result = [
            'Nom'           => $entity->getLastname(),
            'Prenom'        => $entity->getFirstname(),
            'Civilite'      => $entity->getPrefix(),
            'Adresse1'      => $entity->getBillingAddress()->getStreetOne(),
            'Adresse2'      => $entity->getBillingAddress()->getStreetTwo(),
            'CodePostal'    => $entity->getBillingAddress()->getPostcode(),
            'Ville'         => $entity->getBillingAddress()->getCity(),
            'Pays'          => $entity->getBillingAddress()->getCountryId(),
            'DateNaissance' => $entity->getDob(),
            'Telephone'     => $entity->getBillingAddress()->getPhoneNumber(),
            'Email'         => $entity->getEmailAddress(),
            'Provenance'    => self::DEFAULT_REFERER,
            'Societe'       => $entity->getBillingAddress()->getCompany(),
            'CodeExterne'   => $entity->getMagentoId(),
            'Sexe'          => $entity->getGender(),
            'PubParEMail'   => ($entity->hasSubscribedNewsletter() ? '1' : '0')
        ];

        if ($update) {
            $result['Client'] = $entity->getFastmagId();
        } else {
            $result['Magasin'] = $this->config->getValue(Config::XML_PATH_CONNECT_API_SHOP);
        }

        foreach ($result as $field => $data) {
            if ($data === null || $data === '') {
                unset($result[$field]);
            }
        }

        return (object)['Clients' => [(object)$result]];
    }

    /**
     * Update customer on Fastmag
     *
     * @param int            $magentoId
     * @param CustomerEntity $entity
     *
     * @return void
     *
     * @throws JobException
     */
    protected function updateFastmagCustomer($magentoId, $entity)
    {
        $this->checkFastmagEmail($entity->getFastmagId(), $entity->getEmailAddress());

        $billingAddress = $entity->getBillingAddress();

        if ($billingAddress === null) {
            throw new JobException(__(
                'Unable to create customer #%1 on Fastmag: no billing address found.',
                $entity->getMagentoId()
            ));
        }

        $customerData = $this->setDataRequest($entity, true);

        try {
            $response = $this->api->post('/boa/client/update/index.ips', $customerData);

            if (!array_key_exists('data', $response)) {
                throw new ApiException($this->api->getLastRequest(), 'No data received from Fastmag');
            }
            $response = $response['data'][0];

            if ($response['status'] === 'KO') {
                $message = (array_key_exists('Message', $response) ? $response['Message'] : '');

                throw new ApiException($this->api->getLastRequest(), $message);
            }
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to update customer #%1 on Fastmag through API. Code: %2. Message: %3. Data sent: %4',
                $magentoId,
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }
    }

    /**
     * Check email of the customer on Fastmag, and change it if it's not the same as the one in Magento
     *
     * @param int    $fastmagId
     * @param string $email
     *
     * @return void
     *
     * @throws JobException
     */
    protected function checkFastmagEmail($fastmagId, $email)
    {
        try {
            $sql = 'SELECT Email AS email
                FROM clients
                WHERE Client = ' . $this->getFastmagSqlConnection()->escape($fastmagId);

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if ($rows[0]['email'] !== $email) {
                $this->changeCustomerEmail($fastmagId, $email);
            }
        } catch (Exception $exception) {
            throw new JobException(__(
                'Error when trying to check customer email on Fastmag. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                $fastmagId
            ));
        }
    }

    /**
     * Change customer email on Fastmag
     *
     * @param string $oldEmail
     * @param string $newEmail
     *
     * @return void
     *
     * @throws JobException
     */
    protected function changeCustomerEmail($oldEmail, $newEmail)
    {
        $data = (object)['Emails' => [(object)['AncienMail'  => $oldEmail, 'NouveauMail' => $newEmail]]];

        try {
            $this->api->post('/boa/client/email/index.ips', $data);
        } catch (ApiException $exception) {
            throw new JobException(__(
                'Unable to update the email "%1" on Fastmag through API. Code: %2. Message: %3. Data sent: %3',
                $oldEmail,
                $exception->getCode(),
                $exception->getMessage(),
                $exception->getRequest()
            ));
        }
    }
}
