<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-17
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento;

use DateInterval;
use DateTime;
use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Jobqueue\ToMagentoRepository;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotDeleteException;

/**
 * Class Clean
 *
 * Worker class used for cleaning to Magento jobs finished since more than 45 minutes
 */
class Clean extends Worker
{
    /** @inheritDoc */
    protected $code = 'tomagento_clean';

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var ToMagentoRepository $jobRepository */
    protected $jobRepository;

    /**
     * Clean constructor
     *
     * @param Logger                $logger
     * @param ResourceConnection    $resourceConnection
     * @param FastmagSql            $fastmagSql
     * @param SearchCriteriaBuilder $searchCriteriaBuilder
     * @param ToMagentoRepository   $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ToMagentoRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql);

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        try {
            $limitDatetime = new DateTime();
            $limitDatetime->sub(new DateInterval('PT45M'));
        } catch (Exception $e) {
            return;
        }

        $searchCriteria = $this->searchCriteriaBuilder
            ->addFilter(ToMagentoInterface::STATUS, ToMagentoInterface::STATUS_OK)
            ->addFilter(ToMagentoInterface::PROCESSED_AT, $limitDatetime->format('Y-m-d H:i:s'), 'lt')
            ->create();

        $jobsToDelete = $this->jobRepository->getList($searchCriteria);

        foreach ($jobsToDelete->getItems() as $job) {
            try {
                $this->jobRepository->delete($job);
            } catch (CouldNotDeleteException $e) {
                throw new ProcessException($e->getMessage());
            }
        }

        $this->setLastExecutionDate();
    }
}
