<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-10
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Customer
 *
 * Hydration class used for inserting or updating customers from Fastmag to Magento
 */
class Customer extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_customer';

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @inheritDoc */
    protected $subordinateWorkersCodes = ['tomagento_hydration_customer_address'];

    /**
     * Hydration constructor.
     *
     * @param Logger             $logger
     * @param ResourceConnection $resourceConnection
     * @param FastmagSql         $fastmagSql
     * @param Config             $config
     * @param JobRepository      $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config);

        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $results = $this->getDataFromFastmag();
        } catch (ProcessException $exception) {
            foreach ($this->jobs->getItems() as $job) {
                $job->traceException($exception);
            }
        }

        if (is_array($results)) {
            foreach ($results as $row) {
                $job = $this->getJob($row['customer_id']);

                if ($job !== null) {
                    $job->setHydratedData($row);
                }
            }

            foreach ($this->jobs->getItems() as $job) {
                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            }
        }
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $customersIds = $this->jobs->getColumnValues(Job::CONTENT_ID);

        try {
            $sql = 'SELECT Client AS customer_id, Civilite AS prefix, Sexe AS gender, Nom AS lastname,
                    Prenom AS firstname, Societe AS company, Adresse1 AS street1, Adresse2 AS street2,
                    CodePostal AS postcode, Ville AS city,
                    (SELECT CodeIso FROM pays WHERE pays.Pays = clients.Pays) AS country_code, Telephone AS phone,
                    Portable AS mobile, email, JourNaissance AS birth_day, MoisNaissance AS birth_month,
                    AnneeNaissance AS birth_year, Remise AS discount, Regroupement AS regrouping,
                    PubParEMail AS subscribe_newsletter, Actif AS is_active, ObsClient AS comment,
                    DateCreation AS creation_date
                FROM clients
                WHERE Client IN (' . $this->getFastmagSqlConnection()->escape($customersIds) . ')';

            return $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $exception) {
            throw new ProcessException(
                'Error when hydrating customers. Message: %1. Customers IDs: %2',
                $exception->getMessage(),
                implode(', ', $customersIds)
            );
        }
    }

    /**
     * Returns the job for the current customer ID hydrated
     *
     * @param int $customerFastmagId
     *
     * @return Job
     */
    protected function getJob($customerFastmagId)
    {
        return $this->jobs->getItemByColumnValue(Job::CONTENT_ID, $customerFastmagId);
    }
}
