<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-10
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration\Product\Reset;

use Exception;
use Fastmag\Sync\Api\Data\Rule\StoresellerInterface as StoresellerRule;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\StoresellerRepositoryInterface as StoresellerRuleRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Inventory
 *
 * Hydration class used for reseting products prices from Fastmag to Magento
 */
class Inventory extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product_reset_inventory';

    /** @var StoresellerRuleRepository $storesellerRuleRepository */
    protected $storesellerRuleRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /**
     * Product constructor
     *
     * @param Logger                    $logger
     * @param ResourceConnection        $resourceConnection
     * @param FastmagSql                $fastmagSql
     * @param Config                    $config
     * @param StoresellerRuleRepository $storesellerRepository
     * @param SearchCriteriaBuilder     $searchCriteriaBuilder
     * @param JobRepository             $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        StoresellerRuleRepository $storesellerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config);

        $this->storesellerRuleRepository = $storesellerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        foreach ($this->jobs->getItems() as $job) {
            try {
                $hydratedData = $this->getDataFromFastmag();
                $job->setHydratedData($hydratedData);
            } catch (JobException $exception) {
                $job->traceException($exception);

                $this->logger->error(
                    __(
                        '[Job #%1] Error when reseting inventory levels : %2',
                        $job->getId(),
                        $exception->getMessage()
                    )->render()
                );
            }

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $exception) {
                $job->traceException($exception);
            }
        }
    }

    /**
     * Get all catalog's products prices
     *
     * @return array
     *
     * @throws JobException
     */
    protected function getDataFromFastmag()
    {
        $result = [];

        $searchCriteria = $this->searchCriteriaBuilder->create();
        $storesellerRules = $this->storesellerRuleRepository->getList($searchCriteria);

        foreach ($storesellerRules->getItems() as $rule) {
            $fastmagShop = $rule->getFastmagShop();

            $referenceStocks = $this->getReferenceStocks($rule);
            sort($referenceStocks);

            $referenceStocksConcat = implode('|', $referenceStocks);

            try {
                $stockCondition = 'CodeMag IN (' . $this->getFastmagSqlConnection()->escape($referenceStocks) . ')';
            } catch (Exception $exception) {
                throw new JobException(__('Unable to get Fastmag connection: %1', $exception->getMessage()));
            }

            if (!array_key_exists($fastmagShop, $result)) {
                $result[$fastmagShop] = [];
            }

            if (!array_key_exists($referenceStocksConcat, $result[$fastmagShop])) {
                $sql = '
                SELECT GROUP_CONCAT(DISTINCT s.CodeMag SEPARATOR \'|\') AS reference_stocks,
                       s.BarCode  AS sku,
                       s.Couleur  AS color,
                       s.Taille   AS size,
                       p.Produit  AS fastmag_product_id,
                       SUM(s.Qte) AS total_stock,
                       s.AR AS in_stock
                FROM stock AS s INNER JOIN produits AS p
                    ON s.BarCode = p.BarCode AND s.Couleur = p.Couleur AND s.Taille = p.Taille
                WHERE ' . $stockCondition . '
                GROUP BY s.BarCode, s.Couleur, s.Taille';

                try {
                    $rows = $this->getFastmagSqlConnection()->get($sql);
                } catch (Exception $exception) {
                    throw new JobException(
                        __('Unable to get all inventory data from Fastmag: %1', $exception->getMessage())
                    );
                }

                $ruleResults = $this->processRows($rows);

                $result[$fastmagShop][$referenceStocksConcat] = $ruleResults;
            }
        }

        return $result;
    }

    /**
     * Process rows to reset. The point here is minimize the size taken by the hydrated data
     *
     * @param array $rows
     *
     * @return array
     */
    protected function processRows($rows)
    {
        $result = [];

        $currentSku = '';
        $currentInventoryLevels = [];

        foreach ($rows as $row) {
            if ($currentSku !== $row['sku']) {
                if (count($currentInventoryLevels) !== 0) {
                    $result[$currentSku] = $currentInventoryLevels;
                }

                $currentSku = $row['sku'];
                $currentInventoryLevels = [];
            }

            $currentInventoryLevels[$row['fastmag_product_id']] = [
                'in_stock' => (int)$row['in_stock'],
                'stock'    => (int)$row['total_stock']
            ];
        }

        return $result;
    }

    /**
     * Set all stock references used for synchronization
     *
     * @param StoresellerRule $rule
     *
     * @return array
     */
    protected function getReferenceStocks($rule)
    {
        $result = [];

        if ($rule !== null) {
            $result[] = $rule->getFastmagShop();

            $referencesStock = explode('|', $rule->getReferenceStock());
            foreach ($referencesStock as $stock) {
                if (!in_array($stock, $result, true)) {
                    $result[] = $stock;
                }
            }
        }

        return $result;
    }
}
