<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-11
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;

use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Entity\ToMagento\Product as ProductEntity;
use Fastmag\Sync\Process\Worker\ToMagento\Integration\Product;
use Magento\Catalog\Api\CategoryListInterface;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Api\Data\CategoryInterfaceFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class Save
 *
 * Integration worker for tomagento_integration_product_create and tomagento_integration_product_update jobs
 */
class Save extends Product
{
    /** @var string */
    public const PROCESS_CATEGORY_NAME = 'Produits à traiter';

    /** @var CategoryRepositoryInterface $categoryRepository */
    protected $categoryRepository;

    /** @var CategoryInterfaceFactory $categoryFactory */
    protected $categoryFactory;

    /** @var CategoryListInterface $categoryList */
    protected $categoryList;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /**
     * Save constructor
     *
     * @param Logger                      $logger
     * @param ResourceConnection          $resourceConnection
     * @param FastmagSql                  $fastmagSql
     * @param Config                      $config
     * @param JobRepository               $jobRepository
     * @param CategoryRepositoryInterface $categoryRepository
     * @param CategoryInterfaceFactory    $categoryFactory
     * @param CategoryListInterface       $categoryList
     * @param SearchCriteriaBuilder       $searchCriteriaBuilder
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        JobRepository $jobRepository,
        CategoryRepositoryInterface $categoryRepository,
        CategoryInterfaceFactory $categoryFactory,
        CategoryListInterface $categoryList,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config, $jobRepository);

        $this->categoryRepository = $categoryRepository;
        $this->categoryFactory = $categoryFactory;
        $this->categoryList = $categoryList;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        try {
            $this->initiate();
        } catch (ProcessException $exception) {
            $this->logger->notice($exception->getMessage());
            return;
        }

        $processCategoryId = $this->getProcessCategoryId();

        foreach ($this->jobs->getItems() as $job) {
            if (!$job->isInError()) {
                try {
                    $this->processJob($job);
                } catch (JobException $exception) {
                    $job->traceException($exception);

                    $this->logger->error(
                        __(
                            '[Job #%1] Error on product with Fastmag ID #%2: %3',
                            $job->getId(),
                            $this->getJobEntityId($job),
                            $exception->getMessage()
                        )->render()
                    );
                }

                try {
                    $this->jobRepository->save($job);
                } catch (CouldNotSaveException $exception) {
                    $job->traceException($exception);
                }
            }
        }
    }

    /**
     * Get job's entity ID
     *
     * @param Job $job
     *
     * @return int
     */
    protected function getJobEntityId($job)
    {
        /** @var ProductEntity $entity */
        $entity = $job->getEntity();

        return $entity->getFastmagRef();
    }

    /**
     * Get Fastmag process category ID, and create it if it does not exist yet
     *
     * @return int
     *
     * @throws ProcessException
     */
    protected function getProcessCategoryId()
    {
        $this->searchCriteriaBuilder->addFilter(CategoryInterface::KEY_NAME, self::PROCESS_CATEGORY_NAME);

        $searchCriteria = $this->searchCriteriaBuilder->create();

        $categories = $this->categoryList->getList($searchCriteria);

        if ($categories->getTotalCount() > 0) {
            $processCategory = $categories->getItems()[0];

            $processCategoryId = $processCategory->getId();
        } else {
            $processCategory = $this->categoryFactory->create();

            $processCategory->setName(self::PROCESS_CATEGORY_NAME)
                ->setIncludeInMenu(false)
                ->setIsActive(false)
                ->setParentId(1);

            try {
                $this->categoryRepository->save($processCategory);

                $processCategoryId = $processCategory->getId();
            } catch (CouldNotSaveException $e) {
                throw new ProcessException($e->getMessage());
            }
        }

        return $processCategoryId;
    }
}
