<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-29
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection;

use Fastmag\Sync\Api\LogApiRepositoryInterface as LogApiRepository;
use Fastmag\Sync\Exception\ApiException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\LogApiFactory;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\HTTP\ZendClient;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Webapi\Rest\Request;
use Zend_Http_Client_Exception;
use Zend_Http_Response;
use Zend_Uri_Exception;
use Zend_Uri_Http;

/**
 * Class Api
 *
 * Class encapsulating the Fastmag API-related method
 */
class Api
{
    /** @var string */
    protected const API_APPSOURCE = 'Web';

    /** @var string */
    protected const API_VERSION = '1';

    /** @var Config $config */
    protected $config;

    /** @var ZendClient $client */
    protected $client;

    /** @var Zend_Uri_Http $uri */
    protected $uri;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var LogApiRepository $logApiRepository */
    protected $logApiRepository;

    /** @var LogApiFactory $logApiFactory */
    protected $logApiFactory;

    /** @var Logger $logger */
    protected $logger;

    /** @var string $host */
    protected $host;

    /** @var string $chain */
    protected $chain;

    /** @var string $shop */
    protected $shop;

    /** @var string $user */
    protected $user;

    /** @var string $password */
    protected $password;

    /** @var string $token */
    protected $token;

    /**
     * Api constructor
     *
     * @param Config           $config
     * @param ZendClient       $client
     * @param Json             $jsonSerializer
     * @param LogApiRepository $logApiRepository
     * @param LogApiFactory    $logApiFactory
     * @param Logger           $logger
     *
     * @throws ApiException
     */
    public function __construct(
        Config $config,
        ZendClient $client,
        Json $jsonSerializer,
        LogApiRepository $logApiRepository,
        LogApiFactory $logApiFactory,
        Logger $logger
    ) {
        $this->config = $config;
        $this->client = $client;
        $this->jsonSerializer = $jsonSerializer;
        $this->logApiRepository = $logApiRepository;
        $this->logApiFactory = $logApiFactory;
        $this->logger = $logger;

        try {
            $this->uri = Zend_Uri_Http::factory('https');
        } catch (Zend_Uri_Exception $e) {
            throw new ApiException('', 'Unable to create request handler.', 0);
        }
    }

    /**
     * Set API access credentials
     *
     * @param string $host
     * @param string $chain
     * @param string $shop
     * @param string $user
     * @param string $password
     *
     * @return $this
     */
    public function setCredentials($host, $chain, $shop, $user, $password)
    {
        $this->uri->setHost($host);

        $this->chain = $chain;
        $this->shop = $shop;
        $this->user = $user;
        $this->password = $password;

        return $this;
    }

    /**
     * Set credentials from config
     *
     * @return $this
     */
    public function setCredentialsFromConfig()
    {
        $host = $this->config->getValue(Config::XML_PATH_CONNECT_API_HOST);
        $chain = $this->config->getValue(Config::XML_PATH_CONNECT_API_CHAIN);
        $shop = $this->config->getValue(Config::XML_PATH_CONNECT_API_SHOP);
        $user = $this->config->getValue(Config::XML_PATH_CONNECT_API_USER);
        $password = $this->config->getValue(Config::XML_PATH_CONNECT_API_PASSWORD);

        $this->setCredentials($host, $chain, $shop, $user, $password);

        return $this;
    }

    /**
     * Check if the credentials are well filled
     *
     * @return bool
     */
    public function areCredentialsFilled()
    {
        return $this->uri->getHost() !== false
            && $this->chain !== null
            && $this->shop !== null
            && $this->user !== null
            && $this->password !== null;
    }

    /**
     * Returns authentication token
     *
     * @throws ApiException
     *
     * @return bool
     */
    public function authenticate()
    {
        if (!$this->areCredentialsFilled()) {
            $this->setCredentialsFromConfig();
        }

        try {
            $this->client->setUri($this->getUriObject('/boa/auth/token/index.ips'))
                ->setMethod(Request::HTTP_METHOD_POST)
                ->setHeaders(['Content-Type' => 'application/x-www-form-urlencoded'])
                ->setParameterPost([
                    'db'            => $this->chain,
                    'store'         => $this->shop,
                    'Appsource'     => self::API_APPSOURCE,
                    'user'          => $this->user,
                    'pass'          => $this->password,
                    'FastmagApiVer' => self::API_VERSION,
                ]);
        } catch (Zend_Http_Client_Exception $e) {
            throw new ApiException(
                '',
                'Unable to get authentification token for the API',
                0
            );
        }

        $response = $this->client->request();

        $responseData = $this->getResponseData($response);

        if (array_key_exists('data', $responseData) && array_key_exists('token', $responseData['data'])) {
            $this->token = $responseData['data']['token'];
        }

        $this->client->resetParameters();

        return true;
    }

    /**
     * Send a POST query to the API
     *
     * @param string $endpoint
     * @param mixed  $body
     *
     * @return mixed
     *
     * @throws ApiException
     */
    public function post($endpoint, $body)
    {
        if ($this->token === null) {
            $this->authenticate();

            if ($this->token === null) {
                throw new ApiException(
                    $this->getLastRequest(),
                    'Unable to get authentification token for the API',
                    0
                );
            }
        }

        try {
            $this->client->setUri($this->getUriObject($endpoint))
                ->setMethod(Request::METHOD_POST)
                ->setHeaders(['Content-Type' => 'application/json'])
                ->setParameterGet([
                    'db'            => $this->chain,
                    'store'         => $this->shop,
                    'token'         => $this->token,
                    'Appsource'     => self::API_APPSOURCE,
                    'FastmagApiVer' => self::API_VERSION,
                ]);
        } catch (Zend_Http_Client_Exception $exception) {
            throw new ApiException(
                '',
                'Unable to send the request to the API',
                0
            );
        }

        $this->client->setRawData($this->jsonSerializer->serialize($body));

        $response = $this->send();

        $this->client->resetParameters();

        return $this->getResponseData($response);
    }

    /**
     * Send the request, and log it if the configuration is set
     *
     * @return Zend_Http_Response
     */
    public function send()
    {
        $response = $this->client->request();

        if ($this->config->isSetFlag(Config::XML_PATH_LOG_API_CALLS)) {
            $logApi = $this->logApiFactory->create();

            $logApi->setEndpoint($this->client->getUri(true))
                ->setRequest($this->getLastRequest());

            if ($this->client->getLastResponse() !== null) {
                $logApi->setResponse($this->client->getLastResponse()->getBody());
            }

            try {
                $this->logApiRepository->save($logApi);
            } catch (CouldNotSaveException $e) {
                $this->logger->warning('Error when trying to log API call: '. $e->getMessage());
            }
        }

        return $response;
    }

    /**
     * Get last API request
     *
     * @return string
     */
    public function getLastRequest()
    {
        return $this->client->getLastRequest();
    }

    /**
     * Get URI object
     *
     * @param string $path
     *
     * @return Zend_Uri_Http
     *
     * @throws ApiException
     */
    protected function getUriObject($path)
    {
        $this->uri->validateHost($this->host);

        try {
            $this->uri->setPath($path);
        } catch (Zend_Uri_Exception $e) {
            throw new ApiException('', 'The endpoint "' . $path . '" is invalid', 0);
        }

        return $this->uri;
    }

    /**
     * Get response data
     *
     * @param Zend_Http_Response $response
     *
     * @return mixed
     *
     * @throws ApiException
     */
    protected function getResponseData($response)
    {
        $responseBody = $response->getBody();
        $responseJson = $this->jsonSerializer->unserialize($responseBody);

        if (array_key_exists('status', $responseJson) && $responseJson['status'] === 'KO') {
            throw new ApiException(
                $this->getLastRequest(),
                $responseJson['message'],
                $responseJson['errorCode']
            );
        }

        return $responseJson;
    }
}
