<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-22
 ******************************************************************************/

namespace Fastmag\Sync\Model\System\Connection\Sql;

use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Model\System\Connection\Edi;

/**
 * Class EdiSql
 *
 * Class to connect and query to Fastmag through EDI
 */
class EdiSql extends Edi implements SqlInterface
{
    /** @inheritDoc */
    protected $endpoint = 'ediquery.ips';

    /**
     * @inheritDoc
     */
    public function connect()
    {
        $url = $this->getUrl(
            $this->config->getFormattedConfig(),
            [
                'compte'   => $this->config->getUser(),
                'motpasse' => $this->config->getPassword(),
                'enseigne' => $this->config->getChain(),
                'magasin'  => $this->config->getShop(),
                'data'     => 'SELECT NOW()'
            ]
        );

        if ($url === null) {
            throw new NoConnectionException(__('No credentials set in config')->render());
        }

        $this->curlClient->get($url);
        $response = $this->curlClient->getBody();

        if (strpos($response, 'KO|') !== false) {
            throw new NoConnectionException($response);
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function get($sql)
    {
        $result = [];

        $curlOptions = [
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true
        ];

        $this->postCurl($this->getUrl(), trim($sql), [], $curlOptions);
        $response = $this->getResponseBodyCurl();

        if ($response !== false) {
            $lines = explode("\n", $response);

            if (strpos($response, 'KO|') !== false) {
                if (strpos($response, 'KO|ERREUR') !== false) {
                    throw new NoConnectionException($response);
                }
            } elseif (strpos($response, 'Your EDI account has been blocked') !== false) {
                throw new NoConnectionException($response);
            } else {
                foreach ($lines as $line) {
                    $line = str_replace(['<CRLF>', "\r"], ["\n", ''], $line);

                    if (trim($line) !== '') {
                        $row = explode("\t", $line);
                        $result[] = $row;
                    }
                }

                array_walk($result, static function (&$array) use ($result) {
                    $array = @array_combine($result[0], $array);
                });

                // Remove column line
                array_shift($result);
            }
        }

        return $result;
    }

    /**
     * @inheritDoc
     *
     * @todo
     */
    public function post($sql, $params = [])
    {
        $result = $this->get($sql);

        return is_array($result);
    }

    /**
     * @inheritDoc
     *
     * @todo
     */
    public function multiPost($sql)
    {
        return $this->post($sql);
    }

    /**
     * @inheritDoc
     */
    public function escape($value, $forceString = false)
    {
        if (is_array($value)) {
            $result = '';

            foreach ($value as $row) {
                $result .= $this->escape($row, $forceString) . ', ';
            }

            $result = trim($result, ', ');
        } else {
            $result = stripslashes($value);

            if ($forceString || !is_numeric($result)) {
                $search = ['\\', "\0", "\n", "\r", "\x1a", "'", '"'];
                $replace = ['\\\\', '\\0', '\\n', '\\r', "\Z", "\'", '\"'];

                $result = '\'' . strip_tags(nl2br(str_replace($search, $replace, $result))) . '\'';
            }
        }

        return $result;
    }
}
