<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-05-21
 ******************************************************************************/

namespace Fastmag\Sync\Process\Entity\ToFastmag;

use DateTime;
use Exception;
use Fastmag\Sync\Process\EntityInterface;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Gender;
use Fastmag\Sync\Process\Entity\Helper\Attribute\Prefix;
use IntlDateFormatter;

/**
 * Class Customer
 *
 * Entity class for customer.
 * Save and transform data from Magento for Fastmag easy integration,
 * and transfer it from hydration worker to integration worker
 */
class Customer implements EntityInterface
{
    /** @var int $magentoId */
    protected $magentoId;

    /** @var int $fastmagId */
    protected $fastmagId;

    /** @var string $storeId */
    protected $storeId;

    /** @var string $gender */
    protected $gender;

    /** @var string $prefix */
    protected $prefix;

    /** @var string $firstname */
    protected $firstname;

    /** @var string $lastname */
    protected $lastname;

    /** @var string $emailAddress */
    protected $emailAddress;

    /** @var string $dob */
    protected $dob;

    /** @var IntlDateFormatter $dateFormatter */
    protected $dateFormatter;

    /** @var string $subscribeNewsletter */
    protected $subscribeNewsletter;

    /** @var string $createdAt */
    protected $createdAt;

    /** @var Address $billingAddress */
    protected $billingAddress;

    /** @var Address $shippingAddress */
    protected $shippingAddress;

    public function __construct()
    {
        $this->dateFormatter = new IntlDateFormatter(
            'fr_FR',
            IntlDateFormatter::SHORT,
            IntlDateFormatter::NONE,
            'Europe/Paris',
            IntlDateFormatter::GREGORIAN,
            'dd/MM/yyyy'
        );
    }

    /**
     * Get Magento ID
     *
     * @return int
     */
    public function getMagentoId()
    {
        return $this->magentoId;
    }

    /**
     * Get Fastmag ID
     *
     * @return int
     */
    public function getFastmagId()
    {
        return $this->fastmagId;
    }

    /**
     * Get Magento ID
     *
     * @return int
     */
    public function getStoreId()
    {
        return $this->storeId;
    }

    /**
     * Get gender
     *
     * @return string
     */
    public function getGender()
    {
        return Gender::getFastmagValue($this->gender);
    }

    /**
     * Get prefix
     *
     * @return string
     */
    public function getPrefix()
    {
        return Prefix::getFastmagValue($this->prefix);
    }

    /**
     * Get firstname
     *
     * @return string
     */
    public function getFirstname()
    {
        return $this->firstname;
    }

    /**
     * Get lastname
     *
     * @return string
     */
    public function getLastname()
    {
        return $this->lastname;
    }

    /**
     * Get email address
     *
     * @return string
     */
    public function getEmailAddress()
    {
        return strtolower($this->emailAddress);
    }

    /**
     * Get date of birth
     *
     * @return string
     */
    public function getDob()
    {
        return $this->dateFormatter->format($this->dob);
    }

    /**
     * Get if the customer is a subscriber of the newsletter
     *
     * @return bool
     */
    public function hasSubscribedNewsletter()
    {
        return $this->subscribeNewsletter;
    }

    /**
     * Get creation date time as string
     *
     * @return string
     */
    public function getCreatedAt()
    {
        return $this->createdAt;
    }

    /**
     * Get billing address
     *
     * @return Address
     */
    public function getBillingAddress()
    {
        return $this->billingAddress;
    }

    /**
     * Get shipping address
     *
     * @return Address
     */
    public function getShippingAddress()
    {
        return $this->shippingAddress;
    }

    /**
     * Set Magento ID
     *
     * @param int $magentoId
     *
     * @return Customer
     */
    public function setMagentoId($magentoId)
    {
        $this->magentoId = $magentoId;

        return $this;
    }

    /**
     * Set Fastmag ID
     *
     * @param int $fastmagId
     *
     * @return Customer
     */
    public function setFastmagId($fastmagId)
    {
        $this->fastmagId = $fastmagId;

        return $this;
    }

    /**
     * Set store ID
     *
     * @param int $storeId
     *
     * @return Customer
     */
    public function setStoreId($storeId)
    {
        $this->storeId = $storeId;

        return $this;
    }

    /**
     * Set gender
     *
     * @param int $gender
     *
     * @return Customer
     */
    public function setGender($gender)
    {
        $this->gender = $gender;

        return $this;
    }

    /**
     * Set prefix
     *
     * @param string $prefix
     *
     * @return Customer
     */
    public function setPrefix($prefix)
    {
        $this->prefix = $prefix;

        return $this;
    }

    /**
     * Set firstname
     *
     * @param string $firstname
     *
     * @return Customer
     */
    public function setFirstname($firstname)
    {
        $this->firstname = $firstname;

        return $this;
    }

    /**
     * Set lastname
     *
     * @param string $lastname
     *
     * @return Customer
     */
    public function setLastname($lastname)
    {
        $this->lastname = $lastname;

        return $this;
    }

    /**
     * Set email address
     *
     * @param string $emailAddress
     *
     * @return Customer
     */
    public function setEmailAddress($emailAddress)
    {
        $this->emailAddress = $emailAddress;

        return $this;
    }

    /**
     * Set DOB
     *
     * @param string $dob
     *
     * @return Customer
     *
     * @throws Exception
     */
    public function setDob($dob)
    {
        $this->dob = new DateTime($dob);

        return $this;
    }

    /**
     * Set subscribed newsletter
     *
     * @param bool $subscribeNewsletter
     *
     * @return Customer
     */
    public function setSubscribeNewsletter($subscribeNewsletter)
    {
        $this->subscribeNewsletter = $subscribeNewsletter;

        return $this;
    }

    /**
     * Set creation date
     *
     * @param string $createdAt
     *
     * @return Customer
     */
    public function setCreatedAt($createdAt)
    {
        $this->createdAt = $createdAt;

        return $this;
    }

    /**
     * Set billing address
     *
     * @param Address $address
     *
     * @return Customer
     */
    public function setBillingAddress($address)
    {
        $this->billingAddress = $address;

        return $this;
    }

    /**
     * Set shipping address
     *
     * @param Address $address
     *
     * @return Customer
     */
    public function setShippingAddress($address)
    {
        $this->shippingAddress = $address;

        return $this;
    }

    /**
     * @inheritDoc
     */
    public function export()
    {
        return [
            'magento_id'       => $this->getMagentoId(),
            'fastmag_id'       => $this->getFastmagId(),
            'store_id'         => $this->getStoreId(),
            'gender'           => $this->getGender(),
            'prefix'           => $this->getPrefix(),
            'firstname'        => $this->getFirstname(),
            'lastname'         => $this->getLastname(),
            'email_address'    => $this->getEmailAddress(),
            'dob'              => $this->dateFormatter->format($this->getDob()),
            'subscribe_nl'     => $this->hasSubscribedNewsletter(),
            'created_at'       => $this->getCreatedAt(),
            'billing_address'  => ($this->getBillingAddress() ? $this->getBillingAddress()->export() : null),
            'shipping_address' => ($this->getShippingAddress() ? $this->getShippingAddress()->export() : null)
        ];
    }
}
