<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-02-12
 ******************************************************************************/

namespace Fastmag\Sync\Process\Manager;

use Exception;
use Fastmag\Sync\Exception\NoConnectionException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy;
use Fastmag\Sync\Process\Manager;
use Fastmag\Sync\Process\Worker\RemoteSync\Clean;
use Fastmag\Sync\Process\Worker\RemoteSync\Sync;
use Fastmag\Sync\Process\WorkerFactory;
use Magento\Framework\Exception\CouldNotSaveException;

/**
 * Class RemoteSync
 *
 * Sync local queue from remote Fastmag queue
 */
class RemoteSync extends Manager
{
    /** @var int */
    public const SQL_LIMIT = 20000;

    /** @var Proxy $proxy */
    protected $proxy;

    /**
     * RemoteSync constructor.
     *
     * @param Logger        $logger
     * @param Config        $config
     * @param WorkerFactory $workerFactory
     * @param Proxy         $proxy
     */
    public function __construct(
        Logger $logger,
        Config $config,
        WorkerFactory $workerFactory,
        Proxy $proxy
    ) {
        parent::__construct($logger, $config, $workerFactory);

        $this->proxy = $proxy;
    }

    /**
     * Main method
     *
     * @return void
     *
     * @throws ProcessException
     * @throws NoConnectionException
     * @throws Exception
     */
    public function run()
    {
        try {
            $checkOk = $this->checkPrerequisites();

            if ($checkOk) {
                $this->logger->debug('Remote synchronization - Begin process');
                $this->logger->info(
                    'Remote synchronization - Native SQL : ' . ($this->proxy->useConnectionNativeSql() ? 'Yes' : 'No')
                );

                $this->executeWorker('remotesync_instant');
                $this->executeWorker('remotesync_scheduled');

                $this->logger->debug('Remote synchronization - End process');
            }
        } catch (ProcessException $e) {
            $this->logger->critical($e->getMessage());

            throw $e;
        } catch (NoConnectionException $e) {
            $this->logger->alert($e->getMessage());

            throw $e;
        } catch (Exception $e) {
            $this->logger->alert($e->getMessage());

            throw $e;
        }
    }

    /**
     * Check required config fields to run the synchronization
     *
     * @return bool
     *
     * @throws ProcessException
     * @throws NoConnectionException
     */
    protected function checkPrerequisites()
    {
        $result = parent::checkPrerequisites();

        if ($result) {
            if (!$this->proxy->getConnection()) {
                $this->logger->alert('Unable to connect with Fastmag database.');
                $result = false;
            }
        }

        return $result;
    }

    /**
     * Execute worker
     *
     * @param string $workerCode
     *
     * @return void
     *
     * @throws CouldNotSaveException
     */
    protected function executeWorker($workerCode)
    {
        /** @var Sync $syncWorker */
        $syncWorker = $this->workerFactory->create($workerCode);
        $jobsCount = $syncWorker->getJobsCount();

        $this->logger->debug(
            $jobsCount . ' ' . $workerCode . ' events to pull since ' . $syncWorker->getLastExecutionDate()
        );

        $queriesCount = ceil($jobsCount / self::SQL_LIMIT);
        if ($queriesCount) {
            $syncWorker->setQueriesCount($queriesCount)->run();
        } else {
            /** @var Clean $cleanWorker */
            $cleanWorker = $this->workerFactory->create('remotesync_clean');
            $cleanWorker->run();
        }
    }
}
