<?php
/*******************************************************************************
 * Created by HOMEMADE.IO SAS.
 * Fastmag Sync  -  Connect Fastmag with your Magento
 *
 * Copyright (C) HOMEMADE.IO SAS, Inc - All Rights Reserved
 *
 * @author    Simon Laubet-Xavier <simon.laubetxavier@home-made.io>
 * @copyright 2020-2021 HOMEMADE.IO SAS
 * @date      2021-06-10
 ******************************************************************************/

namespace Fastmag\Sync\Process\Worker\ToMagento\Hydration;

use Exception;
use Fastmag\Sync\Api\Data\Jobqueue\ToMagentoInterface as Job;
use Fastmag\Sync\Api\Data\Rule\StoresellerInterface as StoresellerRule;
use Fastmag\Sync\Api\Jobqueue\ToMagentoRepositoryInterface as JobRepository;
use Fastmag\Sync\Api\Rule\StoresellerRepositoryInterface as StoresellerRuleRepository;
use Fastmag\Sync\Exception\JobException;
use Fastmag\Sync\Exception\ProcessException;
use Fastmag\Sync\Logger\Logger;
use Fastmag\Sync\Model\Config;
use Fastmag\Sync\Model\System\Connection\Proxy as FastmagSql;
use Fastmag\Sync\Process\Worker\ToMagento\Hydration;
use InvalidArgumentException;
use Magento\Catalog\Model\ProductFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Api\StoreRepositoryInterface as StoreRepository;

/**
 * Class Product
 *
 * Hydration class used for inserting or updating products from Fastmag to Magento
 */
class Product extends Hydration
{
    /** @inheritDoc */
    protected $code = 'tomagento_hydration_product';

    /** @var ProductFactory $productFactory */
    protected $productFactory;

    /** @var Json $jsonSerializer */
    protected $jsonSerializer;

    /** @var StoresellerRuleRepository $storesellerRuleRepository */
    protected $storesellerRuleRepository;

    /** @var SearchCriteriaBuilder $searchCriteriaBuilder */
    protected $searchCriteriaBuilder;

    /** @var StoreRepository $storeRepository */
    protected $storeRepository;

    /** @var JobRepository $jobRepository */
    protected $jobRepository;

    /** @var Job $currentJob */
    protected $currentJob;

    /** @var StoresellerRule $currentRule */
    protected $currentRule;

    /** @var array $colorsI18n*/
    protected $colorsI18n;

    /** @var array $sizesI18n*/
    protected $sizesI18n;

    /**
     * Product constructor
     *
     * @param Logger                    $logger
     * @param ResourceConnection        $resourceConnection
     * @param FastmagSql                $fastmagSql
     * @param Config                    $config
     * @param ProductFactory            $productFactory
     * @param Json                      $jsonSerializer
     * @param StoresellerRuleRepository $storesellerRepository
     * @param SearchCriteriaBuilder     $searchCriteriaBuilder
     * @param StoreRepository           $storeRepository
     * @param JobRepository             $jobRepository
     */
    public function __construct(
        Logger $logger,
        ResourceConnection $resourceConnection,
        FastmagSql $fastmagSql,
        Config $config,
        ProductFactory $productFactory,
        Json $jsonSerializer,
        StoresellerRuleRepository $storesellerRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        StoreRepository $storeRepository,
        JobRepository $jobRepository
    ) {
        parent::__construct($logger, $resourceConnection, $fastmagSql, $config);

        $this->productFactory = $productFactory;
        $this->jsonSerializer = $jsonSerializer;
        $this->storesellerRuleRepository = $storesellerRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->storeRepository = $storeRepository;
        $this->jobRepository = $jobRepository;
    }

    /**
     * @inheritDoc
     *
     * @throws ProcessException
     */
    public function run()
    {
        $this->getGlobalData();

        foreach ($this->jobs->getItems() as $job) {
            $this->currentJob = $job;

            try {
                $hydratedData = $this->getDataFromFastmag();
                $job->setHydratedData($hydratedData);
            } catch (JobException $exception) {
                $job->traceException($exception);

                $this->logger->error(
                    __(
                        '[Job #%1] Error on product with Fastmag ID #%2: %3',
                        $job->getId(),
                        $job->getContentId(),
                        $exception->getMessage()
                    )->render()
                );
            }

            try {
                $this->jobRepository->save($job);
            } catch (CouldNotSaveException $exception) {
                $job->traceException($exception);
            }
        }
    }

    /**
     * Hydrate the product of the current job
     *
     * @return array
     *
     * @throws JobException
     * @throws ProcessException
     */
    protected function getDataFromFastmag()
    {
        $fastmagRef = $this->currentJob->getContentId();

        if ($this->isPack($fastmagRef)) {
            throw new JobException(__(
                'Product "%1" is a pack, which can not be imported with Fastmag_Sync module',
                $fastmagRef
            ));
        }

        $hydratedData = [
            'colors_18n' => $this->colorsI18n,
            'sizes_18n'  => $this->sizesI18n,
            'parent_ref' => $fastmagRef,
            'is_color'   => false,
            'magento_id' => $this->getMagentoProductId($fastmagRef)
        ];

        $hydratedData = array_merge($hydratedData, $this->getBasicData($fastmagRef));

        if ($this->productMustBeSynced($hydratedData)) {
            $storesList = $this->storeRepository->getList();

            foreach ($storesList as $store) {
                try {
                    $this->currentRule = $this->storesellerRuleRepository->getByStoreId($store->getId());
                } catch (Exception $e) {
                    continue;
                }

                $hydratedDataStore['vat_rate'] = $this->getFastmagShopTaxRate($this->currentRule->getFastmagShop());

                $fastmagStaticData = $this->getStaticData($fastmagRef);
                $hydratedDataStore['children'] = $fastmagStaticData;

                $fastmagI18nData = $this->getI18nData($fastmagRef);
                $hydratedDataStore['i18n'] = $fastmagI18nData;

                $hydratedData[$store->getCode()] = $hydratedDataStore;
            }
        }

        return $hydratedData;
    }

    /**
     * Get global data (independent of the jobs)
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getGlobalData()
    {
        $this->getColorsI18n();
        $this->getSizesI18n();
    }

    /**
     * Get colors labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getColorsI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Couleur AS color, Traduction AS translation FROM couleurslangues';

            $this->colorsI18n = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }
    }

    /**
     * Get sizes labels i18n
     *
     * @return void
     *
     * @throws ProcessException
     */
    protected function getSizesI18n()
    {
        try {
            $sql = 'SELECT Langue AS language, Taille AS size, Traduction AS translation FROM tailleslangues';

            $this->sizesI18n = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }
    }

    /**
     * Check if product is a pack
     *
     * @param string $fastmagRef
     *
     * @return bool
     */
    protected function isPack($fastmagRef)
    {
        return (strpos($fastmagRef, 'PACK_') !== false);
    }

    /**
     * Get Magento product ID for Fastmag ref given
     *
     * @param string $fastmagRef
     *
     * @return int|false
     */
    protected function getMagentoProductId($fastmagRef)
    {
        $product = $this->productFactory->create();

        return $product->getIdBySku($fastmagRef);
    }

    /**
     * Get product basic data on Fastmag (VisibleWeb, Actif, creation date, inventory data, family)
     *
     * @param string $fastmagRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getBasicData($fastmagRef)
    {
        try {
            $stockListCondition = '';

            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $stockList = $this->getReferenceStocks();

                if (!is_array($stockList) || count($stockList) === 0) {
                    throw new ProcessException(
                        'No store/seller rules is configured, please create some rules,'
                        . ' or disable "Only import declinations received by sellers" config field.'
                    );
                }

                $stockListCondition = ' AND s.CodeMag IN (' . $this->getFastmagSqlConnection()->escape($stockList) . ')';
            }

            $sql = 'SELECT pf.VisibleWeb, pf.Actif, pf.DateCreation, COUNT(s.ID) AS Stocks,
                    GROUP_CONCAT(DISTINCT IF(Stock >= 1, s.CodeMag, NULL)) AS CodeMagList, pf.Famille, f.TauxTVA
                FROM produitsfiches AS pf LEFT JOIN stock AS s on pf.BarCode = s.BarCode
                    LEFT JOIN familles AS f ON pf.Famille = f.Famille
                WHERE pf.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                    AND s.AR = 1 ' . $stockListCondition . '
                GROUP BY pf.BarCode';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        $row = reset($rows);

        return [
            'visible_web'   => $row['VisibleWeb'] ?? 0,
            'active'        => $row['Actif'] ?? 0,
            'creation_date' => $row['DateCreation'] ?? '',
            'stock_level'   => $row['Stocks'] ?? 0,
            'stocks_list'   => $row['CodeMagList'] ? explode(',', $row['CodeMagList']) : '',
            'family'        => $row['Famille'] ?? '',
            'vat_rate'      => $row['TauxTVA'] ?? ''
        ];
    }

    /**
     * Check if products fill all the condition to be synced
     *
     * @param array $hydratedData
     *
     * @return bool
     */
    protected function productMustBeSynced($hydratedData)
    {
        $result = ($hydratedData['active'] && $hydratedData['visible_web'])
            || !$this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ACTIVE_VISIBLEWEB);

        $onlyStoreseller = $this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER);
        if ($hydratedData['stock_level'] === 0 && $onlyStoreseller) {
            $result = false;
        }

        if ($hydratedData['magento_id']) {
            $result = true;
        }

        return $result;
    }

    /**
     * Set all stock references used for synchronization
     *
     * @return array
     */
    protected function getReferenceStocks()
    {
        $result = [];

        if ($this->currentRule !== null) {
            $result[] = $this->currentRule->getFastmagShop();

            $referencesStock = explode('|', $this->currentRule->getReferenceStock());
            foreach ($referencesStock as $stock) {
                if (!in_array($stock, $result, true)) {
                    $result[] = $stock;
                }
            }
        }

        return $result;
    }

    /**
     * Get tax rate for the product
     *
     * @param string $fastmagShop
     * @param float  $defaultValue
     *
     * @return float
     *
     * @throws ProcessException
     */
    protected function getFastmagShopTaxRate($fastmagShop, $defaultValue = 20.0)
    {
        $result = $defaultValue;

        try {
            $sql = 'SELECT TVA FROM mag WHERE Code = ' . $this->getFastmagSqlConnection()->escape($fastmagShop);

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        if (count($rows) > 0) {
            foreach ($rows as $row) {
                $result = $row['TVA'];
            }
        }

        return $result;
    }

    /**
     * Get static data for the product from Fastmag
     *
     * @param string $fastmagRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getStaticData($fastmagRef)
    {
        $result = [];

        try {
            $rateCode = $this->currentRule->getRateCode();
            $fastmagShop = $this->currentRule->getFastmagShop();

            $standardPriceSubquery = $this->getStandardPriceSubquery($fastmagRef, $fastmagShop, $rateCode);
            $combinationPriceSubquery = $this->getCombinationPriceSubquery($fastmagRef, $rateCode);
            $priceRateSubquery = $this->getPriceRateSubquery($fastmagRef, $rateCode);
            $salesPriceSubquery = $this->getSalesPriceSubquery($fastmagRef, $rateCode);
            $salesPriceChildrenSubquery = $this->getSalesPriceChildrenSubquery($fastmagShop);
            $buyingPriceSubquery = $this->getBuyingPriceSubquery($fastmagRef);

            $onlyStoresellerCombination = ' AND (no_web < 1 OR no_web IS NULL)';
            if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_IMPORT_ONLY_STORESELLER)) {
                $onlyStoresellerCombination .= ' AND enable_stock = 1 ';
            }

            $stockCondition = '';
            $referenceStocks = $this->getReferenceStocks();
            if ($referenceStocks !== null && count($referenceStocks) !== 0) {
                $stockCondition = ' AND CodeMag IN ('
                    . $this->getFastmagSqlConnection()->escape($referenceStocks) . ')';
            }

            $sql = 'SELECT
                    (
                        SELECT JSON_OBJECT(
                            \'family\', Famille,
                            \'subfamily\', SsFamille,
                            \'season\', Saison,
                            \'section\', Rayon,
                            \'brand\', Marque
                        )
                        FROM produitsfiches AS pf
                        WHERE pf.BarCode = stock.BarCode
                    ) AS rules_data,
                    (
                        SELECT JSON_OBJECT(
                            \'fm_product_id\', Produit,
                            \'ean13\', RIGHT(GenCod, 13),
                            \'supplier_ref\', RefFournisseur,
                            \'weight\', PoidsTC
                        )
                        FROM produits AS p
                        WHERE p.BarCode = stock.BarCode AND p.Taille = stock.Taille AND p.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS combination_data,
                    (
                        SELECT ProduitActif
                        FROM produits AS p2
                        WHERE p2.BarCode = stock.BarCode AND p2.Taille = stock.Taille AND p2.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS enable,
                    (
                        SELECT 1
                        FROM stock AS s2
                        WHERE s2.BarCode = stock.BarCode AND s2.Taille = stock.Taille AND s2.Couleur = stock.Couleur
                            ' . $stockCondition . '
                        LIMIT 1
                    ) AS enable_stock,
                    (
                        SELECT IFNULL(ValeurInteger, -1) AS no_web
                        FROM produits AS p3 LEFT JOIN complementsr AS c
                            ON p3.Produit = c.Clef AND c.Champ = \'NO_WEB\' AND c.Nature = \'PRODUITSTC\'
                        WHERE p3.BarCode = stock.BarCode AND p3.Taille = stock.Taille AND p3.Couleur = stock.Couleur
                        LIMIT 1
                    ) AS no_web,
                    (' . $standardPriceSubquery . ') AS standard_price,
                    (' . $combinationPriceSubquery . ') AS combination_price,
                    (' . $priceRateSubquery . ') AS price_rate,
                    (' . $salesPriceSubquery . ') AS sales_price,
                    (' . $salesPriceChildrenSubquery . ') AS sales_price_children,
                    (' . $buyingPriceSubquery . ') AS buying_price,
                    stock.Taille AS size,
                    stock.Couleur AS color,
                    (
                        SELECT Descriptif
                        FROM produitscouleurs AS pc
                        WHERE pc.Barcode = stock.BarCode AND pc.Couleur = stock.Couleur
                    ) AS alternative_color,
                    SUM(Stock) AS total_stock,
                    Poids AS weight,
                    stock.AR AS in_stock
                FROM produitsfiches AS pf INNER JOIN stock ON pf.BarCode = stock.BarCode
                WHERE stock.AR = 1
                    AND stock.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                    ' . $stockCondition . '
                GROUP BY stock.Barcode, stock.Taille, stock.Couleur
                HAVING enable = 1 ' . $onlyStoresellerCombination . '
                ORDER BY stock.Couleur, stock.Taille';

            $rows = $this->getFastmagSqlConnection()->get($sql);

            if (count($rows) === 0) {
                $rows = $this->getFastmagSqlConnection()->get(str_replace('stock.AR = 1', 'stock.AR = 0', $sql));
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        foreach ($rows as $row) {
            $row = $this->extractJsonData($row);

            if (!array_key_exists($row['color'], $result)) {
                $result[$row['color']] = [$row['size'] => $row];
            } else {
                $result[$row['color']][$row['size']] = $row;
            }
        }

        return $result;
    }

    /**
     * Returns subquery for stock price (standard price)
     *
     * @param string $fastmagRef
     * @param string $fastmagShop
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getStandardPriceSubquery($fastmagRef, $fastmagShop, $priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT stpr.PrixVente
                    FROM stock AS stpr
                    WHERE stpr.CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                        AND stpr.AR = 1
                        AND stpr.Taille = stock.Taille
                        AND stpr.Couleur = stock.Couleur
                        AND stpr.PrixVente > 0
                        AND stpr.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                    ORDER BY stpr.Date, stpr.ID
                    LIMIT 1';
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for combination price
     *
     * @param string $fastmagRef
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getCombinationPriceSubquery($fastmagRef, $priceRateCode)
    {
        try {
            if ($priceRateCode === 'prix_indicatif') {
                $result = 'pf.PrixVente';
            } else {
                $result = 'SELECT p.PrixVente
                    FROM produits AS p
                    WHERE p.PrixVente > 0
                        AND p.Taille = stock.Taille
                        AND p.Couleur = stock.Couleur
                        AND p.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true);
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for combination price
     *
     * @param string $fastmagRef
     * @param string $priceRateCode
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getPriceRateSubquery($fastmagRef, $priceRateCode)
    {
        try {
            // Specific tarif price
            if ($priceRateCode !== '') {
                if ($priceRateCode === 'prix_indicatif') {
                    $result = 'pf.PrixVente';
                } else {
                    $result = 'SELECT pt.Prix
                        FROM produitstarifs AS pt
                        WHERE pt.Tarif = ' . $this->getFastmagSqlConnection()->escape($priceRateCode) . '
                            AND pt.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                            AND pt.Taille = stock.Taille AND pt.Prix > 0 AND pt.Couleur = stock.Couleur
                        ORDER BY pt.Prix desc
                        LIMIT 1';
                }
            } else {
                $result = '0';
            }
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for sales price
     *
     * @param string $fastmagRef
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getSalesPriceSubquery($fastmagRef, $fastmagShop)
    {
        try {
            $result = 'SELECT JSON_OBJECT(
                        \'sales_price\', Prix,
                        \'discount\', Remise,
                        \'begin_at\', DateDebut,
                        \'end_at\', DateFin,
                        \'reason\', Motif
                    )
                FROM prixremise
                WHERE CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                    AND BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                    AND DateDebut <= date(NOW()) AND DateFin >= date(NOW())
                ORDER BY DateDebut DESC, DateFin DESC, Remise, Prix
                LIMIT 1';
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for size and color sales price
     *
     * @param string $fastmagShop
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getSalesPriceChildrenSubquery($fastmagShop)
    {
        try {
            $result = 'SELECT JSON_OBJECT(
                        \'sales_price\', Prix,
                        \'discount\', Remise,
                        \'begin_at\', DateDebut,
                        \'end_at\', DateFin,
                        \'reason\', Motif
                    )
                FROM prixremisetc AS ptc
                WHERE ptc.CodeMag = ' . $this->getFastmagSqlConnection()->escape($fastmagShop) . '
                    AND ptc.BarCode = stock.BarCode AND ptc.Taille = stock.Taille AND ptc.Couleur = stock.Couleur
                    AND (ptc.Remise > 0 OR ptc.Prix > 0) AND DateDebut <= date(NOW()) AND ptc.DateFin >= date(NOW())
                ORDER BY ptc.DateDebut DESC, ptc.DateFin DESC, ptc.Remise, ptc.Prix
                LIMIT 1';
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Returns subquery for buying price
     *
     * @param string $fastmagRef
     *
     * @return string
     *
     * @throws ProcessException
     */
    protected function getBuyingPriceSubquery($fastmagRef)
    {
        try {
            $result = 'SELECT prod.PrixAchat
                FROM produits AS prod
                WHERE prod.Taille = stock.Taille AND prod.Couleur = stock.Couleur
                    AND prod.BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                LIMIT 1';
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return $result;
    }

    /**
     * Extract JSON data and merge them to the row given
     *
     * @param array $row
     *
     * @return array
     */
    protected function extractJsonData($row)
    {
        try {
            $rulesData = $this->jsonSerializer->unserialize($row['rules_data']);
            $row = array_merge($row, $rulesData);
            unset($row['rules_data']);
        } catch (InvalidArgumentException $e) {
            $this->logger->warning(
                '[Job #' . $this->currentJob->getId()
                . '] Error when unserializing rules_data from Fastmag: ' . $row['rules_data']
            );
        }

        try {
            $combinationData = $this->jsonSerializer->unserialize($row['combination_data']);
            $row = array_merge($row, $combinationData);
            unset($row['combination_data']);
        } catch (InvalidArgumentException $e) {
            $this->logger->warning(
                '[Job #' . $this->currentJob->getId()
                . '] Error when unserializing combination_data from Fastmag: ' . $row['combination_data']
            );
        }

        return $row;
    }

    /**
     * Get minimal price (indicative value) from Fastmag, if price can not be synced by other means
     *
     * @param string $fastmagRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getMinimalPrice($fastmagRef)
    {
        try {
            $sql = 'SELECT PrixVente AS PrixVenteIndicatif
                FROM produits
                WHERE BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true) . '
                ORDER BY PrixVente
                LIMIT 1';

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        return reset($rows);
    }

    /**
     * Get all i18nized data from Fastmag
     *
     * @todo add store parameter to actually retrieve i18nized data
     *
     * @param string $fastmagRef
     *
     * @return array
     *
     * @throws ProcessException
     */
    protected function getI18nData($fastmagRef)
    {
        $additionalAttributes = $this->getAdditionalAttributes();

        try {
            $sql = 'SELECT \'FR\' AS language, Designation AS designation, Designation2 AS designation2,
                    Marque AS brand, Fournisseur AS supplier, Lavage AS washing, Chlore AS chlorine,
                    Repassage AS ironing, Pressing AS dry_cleaning, Sechage AS drying, AquaNettoyage AS water_cleaning,
                    Descriptif AS description, DateCreation AS created_at' . $additionalAttributes . '
                FROM produitsfiches
                WHERE BarCode = ' . $this->getFastmagSqlConnection()->escape($fastmagRef, true);

            $rows = $this->getFastmagSqlConnection()->get($sql);
        } catch (Exception $e) {
            throw new ProcessException($e->getMessage());
        }

        $result['FR'] = reset($rows);

        return $result;
    }

    /**
     * Returns SQL to get data for additionnal attributes, given config fields
     *
     * @return string
     */
    protected function getAdditionalAttributes()
    {
        $additionalAttributes = [];

        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_FAMILY)) {
            $additionalAttributes[] = 'Famille AS family';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SUBFAMILY)) {
            $additionalAttributes[] = 'SsFamille AS subfamily';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SECTION)) {
            $additionalAttributes[] = 'Rayon AS section';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_SEASON)) {
            $additionalAttributes[] = 'Saison AS season';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MODEL)) {
            $additionalAttributes[] = 'Modele AS model';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_MATERIAL)) {
            $additionalAttributes[] = 'Matiere AS material';
        }
        if ($this->config->isSetFlag(Config::XML_PATH_PRODUCT_OTHER_ATTRIBUTES_SYNC_THEME)) {
            $additionalAttributes[] = 'Theme AS theme';
        }

        return (count($additionalAttributes) > 0 ? ', ' . implode(', ', $additionalAttributes) : '');
    }
}
